% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/netCDF.R
\name{ncdf4_data_extract}
\alias{ncdf4_data_extract}
\title{Extract data ranges from a NetCDF file format 4}
\usage{
ncdf4_data_extract(cdfFile, massValues, timeRange, useRT = FALSE)
}
\arguments{
\item{cdfFile}{A path to a NetCDF file format 4.}

\item{massValues}{A numeric vector representing m/z values.}

\item{timeRange}{A numeric vector of length 2 representing the lower and upper
time limits.}

\item{useRT}{Logical. If \code{TRUE}, the time range is in seconds, otherwise if \code{FALSE}
(default), the time range is in retention time index units (\code{TRUE}).}
}
\value{
A named list with the following components.
\itemize{
\item \code{Time} Numeric vector: the retention time in seconds
\item \code{Index} Numeric vector: the retention time index (or zero if the file was
was not retention time corrected
\item \code{Intensity} Matrix: Rows are the retention times (or scans) and columns are masses.
\item \code{massRange} Numeric vector of length 2: the mass range of the CDF
}
}
\description{
A flexible and convenient function to extract raw data from a NetCDF file format
4 using time ranges and m/z ranges or values. This is a better (and faster)
alternative to the old \code{\link[=peakCDFextraction]{peakCDFextraction()}} function, which reads the whole CDF
into memory, specially when only sections of the CDF are needed.
}
\details{
The function takes a NetCDF format 4 generated by "TargetSearch" and extracts
raw intensity values from given m/z ion traces within a given time range. The
time range can be in seconds or arbitrary retention time index units. For
the latter case, the function expects a time corrected CDF file.

If the given time range is out of range, a \code{NULL} value will be returned. In contrast,
if the m/z values are out of range, then zeros will be returned for out of range masses
(provided that the time range is not out of range). If \code{timeRange} is missing, then
the whole time range. Similarly, if \code{massRange} is missing, then all the masses are
extracted. If both are missing, the function behaves as \code{\link[=peakCDFextraction]{peakCDFextraction()}}, but
the output (a named list) uses slightly different names.

The NetCDF must be have been previously converted to the custom "TargetSearch" format,
otherwise an error will be raised. See \code{\link[=ncdf4_convert]{ncdf4_convert()}} for the conversion.
}
\note{
An error will be produced for invalid files. Also, this function is intended
to be used internally, but it is exposed for convenience, for example, to
create custom plots.
}
\examples{
# If you already have a NCDF4 file set the variable accordingly
# and skip the following section
# nc4file <- "/path/to/netcdf4.nc4"

### create NCDF4 from NCDF3 from the package TargetSearchData
require(TargetSearchData)
# pick first file
cdffile <- tsd_cdffiles()[1]
nc4file <- tempfile(fileext=".nc4")
ncdf4_convert(cdffile, nc4file)

# update RI data using pre-computed values
ncdf4_update_ri(nc4file, c(252, 311, 269), c(262320, 323120, 381020))
### end

# extract all data (behaves like peakCDFextraction)
data <- ncdf4_data_extract(nc4file)

# extract only certain m/z values
data <- ncdf4_data_extract(nc4file, massValues=c(116, 192))

# to use mass ranges, use the colon (:) operator for example
data <- ncdf4_data_extract(nc4file, massValues=c(120:130, 200, 203:209))

# restrict extraction to a retention index interval
data <- ncdf4_data_extract(nc4file, massValues=c(116, 192),
                           timeRange=c(200000, 220000))

# same, but using retention time in seconds.
data <- ncdf4_data_extract(nc4file, massValues=c(116, 192),
                           timeRange=c(200, 300), useRT=TRUE)
}
\seealso{
\code{\link[=ncdf4_convert]{ncdf4_convert()}}, \code{\link[=peakCDFextraction]{peakCDFextraction()}}
}
\author{
Alvaro Cuadros-Inostroza
}
