% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/main.R
\name{rasterizeMatrix}
\alias{rasterizeMatrix}
\title{rasterizeMatrix}
\usage{
rasterizeMatrix(
  data,
  pos,
  bbox,
  resolution = 100,
  square = TRUE,
  fun = "mean",
  n_threads = 1,
  BPPARAM = NULL,
  verbose = TRUE
)
}
\arguments{
\item{data}{\code{dgCmatrix} or \code{matrix}: Feature x observation matrix represented as a 
\code{dgCmatrix} or \code{matrix} object. Features can be genes or cell types. In the case of 
features being cell types, this matrix is assumed to be a sparse model matrix 
with rows as cell types and columns as cell IDs.}

\item{pos}{\code{matrix}: Spatial x,y coordinates of observations stored as a 
matrix array. Further, x,y coordinates are assumed to be stored in column 1 
and 2 of \code{spatialCoords}.}

\item{bbox}{\code{bbox} or \code{numeric}: Bounding box for rasterization defined 
by a bbox class object (as created by sf::st_bbox) or a numeric vector of length four, with xmin, ymin, xmax 
and ymax values. Values in a numeric vector are assumed to be in the order of xmin, 
ymin, xmax, and ymax.}

\item{resolution}{\code{integer} or \code{double}: Resolution refers to the side 
length of each pixel for square pixels and the distance between opposite edges 
of each pixel for hexagonal pixels. The unit of this parameter is assumed to 
be the same as the unit of spatial coordinates of the input data.}

\item{square}{\code{logical}: If TRUE (default), rasterize into square pixels. If FALSE, 
rasterize into hexagonal pixels.}

\item{fun}{\code{character}: If "mean", pixel value for each pixel 
would be the average of gene expression for all cells within the pixel. If 
"sum", pixel value for each pixel would be the sum of gene expression for all 
cells within the pixel.}

\item{n_threads}{\code{integer}: Number of threads for parallelization. Default = 1. 
Inputting this argument when the \code{BPPARAM} argument is missing would set parallel 
exeuction back-end to be \code{BiocParallel::MulticoreParam(workers = n_threads)}. 
We recommend setting this argument to be the number of cores available 
(\code{parallel::detectCores(logical = FALSE)}). If \code{BPPARAM} argument is 
not missing, the \code{BPPARAM} argument would override \code{n_threads} argument.}

\item{BPPARAM}{\code{BiocParallelParam}: Optional additional argument for parallelization. 
This argument is provided for advanced users of \code{BiocParallel} for further 
flexibility for setting up parallel-execution back-end. Default is NULL. If 
provided, this is assumed to be an instance of \code{BiocParallelParam}.}

\item{verbose}{\code{logical}: Whether to display verbose output or warning. Default is FALSE}
}
\value{
The output is returned as a \code{list} containing rasterized feature 
x observation matrix as \code{dgCmatrix} if data was given as \code{dgCmatrix} 
and as \code{matrix} if data was given as \code{matrix}, spatial x,y coordinates of pixel 
centroids as \code{matrix}, and \code{data.frame} containing meta data for pixels 
(number of cells that were aggregated in each pixel, cell IDs of cells that were 
aggregated in each pixel, pixel type based on the \code{square} argument, pixel 
resolution based on the \code{resolution} argument, pixel geometry as \code{sfc_POLYGON}).
}
\description{
Function to rasterize a given input matrix (both dense or sparse) based on a given position matrix.

This function assumes that inputs are provided as a \code{dgCmatrix} 
or \code{matrix} for data and \code{matrix} for position.
}
\examples{
library(SpatialExperiment)
library(sf)

data("merfish_mousePOA")
# extract features-by-cells matrix, spatial coordinates from the SpatialExperiment object
data <- assay(merfish_mousePOA)
pos <- spatialCoords(merfish_mousePOA)
# compute bounding box
resolution <- 100
bbox <- st_bbox(c(
  xmin = floor(min(pos[,1])-resolution/2), 
  xmax = ceiling(max(pos[,1])+resolution/2), 
  ymin = floor(min(pos[,2])-resolution/2), 
  ymax = ceiling(max(pos[,2])+resolution/2)
))
# rasterize with mean as the aggregation function
out_mean <- rasterizeMatrix(data, pos, bbox, resolution = resolution, fun = "mean")
# rasterize with sum as the aggregation function
out_sum <- rasterizeMatrix(data, pos, bbox, resolution = resolution, fun = "sum")
# rasterize with user-defined resolution and hexagonal pixels
# in this case, you need to update the bbox as well
resolution <- 200
bbox <- st_bbox(c(
  xmin = floor(min(pos[,1])-resolution/2), 
  xmax = ceiling(max(pos[,1])+resolution/2), 
  ymin = floor(min(pos[,2])-resolution/2), 
  ymax = ceiling(max(pos[,2])+resolution/2)
))
out_hex <- rasterizeMatrix(data, pos, bbox, resolution = resolution, square = FALSE, fun = "mean")

}
