% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/samples.R
\name{associateComponents}
\alias{associateComponents}
\title{Runs linear models for components and sample-level data}
\usage{
associateComponents(
  re,
  formula,
  method = "lm",
  scale_reduced = TRUE,
  center_reduced = TRUE,
  type = "II",
  adj_method = "BH",
  ...
)
}
\arguments{
\item{re}{An object inheriting from
\link[ReducedExperiment]{ReducedExperiment}.}

\item{formula}{The model formula to apply. Only the right hand side of the
model need be specified (e.g., "~ x + y"). The left hand side (outcome)
represents the
components themselves. The variables in this formula should be present in the
\code{colData} of \code{re}.}

\item{method}{If "lm", then the \link[stats]{lm} function is used to run
linear models (in tandem with \link[car]{Anova} for running anovas on the
model terms). If "lmer", then linear mixed models are run through
\link[lmerTest]{lmer}.}

\item{scale_reduced}{If \code{TRUE}, the reduced data are scaled (to have a standard
deviation of 1) before modelling.}

\item{center_reduced}{If \code{TRUE}, the reduced data are centered (to have a mean
of 0) before modelling.}

\item{type}{The type of anova to be applied to the terms of the linear model.}

\item{adj_method}{The method for adjusting for multiple testing. Passed to
the \link[stats]{p.adjust} \code{method} parameter.}

\item{...}{Additional arguments passed to \link[lmerTest]{lmer}, given that
\code{method} is set to "lmer".}
}
\value{
Returns a list with the entry "models" including a list of the
model objects, "anovas" containing the output of anova-based testing,
and "summaries" containing the results of running \code{summary} on the models.
}
\description{
Runs either standard linear or linear mixed models, with reduced components
(e.g., factors or modules) as the outcomes  and sample-level information
(e.g., treatment, disease status) as predictors.
}
\details{
Multiple testing adjustment is performed separately for each term in the
model across all factors.
In other words, p-values are adjusted for the number of factors, but not
the number of model terms. If you are testing a large number of terms,
you could consider applying a custom adjustment method or using
penalised regression.
}
\examples{
# Create FactorisedExperiment with random data (100 features, 50 samples,
# 10 factors)
set.seed(1)
fe <- ReducedExperiment:::.createRandomisedFactorisedExperiment(100, 50, 10)
fe

# Create a sample-level variable describing some sort of treatment
colData(fe)$treated <- c(rep("control", 25), rep("treatment", 25))
colData(fe)$treated <- factor(colData(fe)$treated, c("control", "treatment"))

# Increase the value of factor 1 for the treated samples, simulating some
# kind of treatment-related effect picked up by factor analysis
reduced(fe)[, 1][colData(fe)$treated == "treatment"] <-
    reduced(fe)[, 1][colData(fe)$treated == "treatment"] +
    rnorm(25, mean = 1.5, sd = 0.1)

# Create a sample-level variable describing a covariate we want to adjust for
# We will make the treated patients slightly older on average
colData(fe)$age <- 0
colData(fe)$age[colData(fe)$treated == "control"] <- rnorm(25, mean = 35, sd = 8)
colData(fe)$age[colData(fe)$treated == "treatment"] <- rnorm(25, mean = 40, sd = 8)

# Associate the factors with sample-level variable in the colData
lm_res <- associateComponents(
    fe,
    formula = "~ treated + age", # Our model formula
    method = "lm", # Use a linear model
    adj_method = "BH" # Adjust our p-values with Benjamini-Hochberg
)

# We see that treatment is significantly associated with factor 1 (adjusted
# p-value < 0.05) and is higher in the treated patients. Age is not
# significantly associated with factor 1, but there is a slight positive
# relationship
print(head(lm_res$summaries[
    ,
    c("term", "component", "estimate", "stderr", "pvalue", "adj_pvalue")
]))

# But what if these aren't 50 independent patients, but rather 25 patients
# sampled before and after treatment? We can account for this using a
# linear mixed model, which can account for repeated measures and paired
# designs

# First we add in this information
colData(fe)$patient_id <- c(paste0("patient_", 1:25), paste0("patient_", 1:25))

# Then we run the linear mixed model with a random intercept for patient
lmm_res <- associateComponents(
    fe,
    formula = "~ treated + age + (1 | patient_id)", # Add a random intercept
    method = "lmer", # Use a linear mixed model
    adj_method = "BH"
)

# We used a different method, but can obtain a similar summary output
print(head(lmm_res$summaries[
    ,
    c("term", "component", "estimate", "stderr", "pvalue", "adj_pvalue")
]))

}
\seealso{
\code{\link[stats:lm]{stats::lm()}}, \code{\link[car:Anova]{car::Anova()}}, \code{\link[lmerTest:lmer]{lmerTest::lmer()}}
}
\author{
Jack Gisby
}
