% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plotSpectra-PTM.R
\name{plotSpectraPTM}
\alias{plotSpectraPTM}
\title{Function to plot MS/MS spectra with PTMs}
\usage{
plotSpectraPTM(
  x,
  deltaMz = TRUE,
  ppm = 20,
  xlab = "m/z",
  ylab = "intensity [\%]",
  xlim = numeric(),
  ylim = numeric(),
  main = character(),
  col = c(y = "darkred", b = "darkblue", acxy = "darkgreen", other = "grey40"),
  labelCex = 1,
  labelSrt = 0,
  labelAdj = NULL,
  labelPos = 3,
  labelOffset = 0.5,
  asp = 1,
  minorTicks = TRUE,
  USI = TRUE,
  ...
)
}
\arguments{
\item{x}{a \code{Spectra()} object.}

\item{deltaMz}{\code{logical(1L)} If \code{TRUE}, adds an additional plot showing the
difference of mass over charge between matched oberved and theoretical
fragments in parts per million. Does not yet support modifications. The
matching is based on \code{calculateFragments()} and needs a 'sequence'
variable in \code{spectraVariables(x)}. Default is set to \code{TRUE}.}

\item{ppm}{\code{integer(1L)} Sets the limits of the delta m/z plot and is passed
to \code{labelFragments()}.}

\item{xlab}{\code{character(1)} with the label for the x-axis (by default \code{xlab = "m/z"}).}

\item{ylab}{\code{character(1)} with the label for the y-axis (by default \code{ylab = "intensity"}).}

\item{xlim}{\code{numeric(2)} defining the x-axis limits. The range of m/z values
are used by default.}

\item{ylim}{\code{numeric(2)} defining the y-axis limits. The range of intensity
values are used by default.}

\item{main}{\code{character(1)} with the title for the plot. By default the
spectrum's MS level and retention time (in seconds) is used.}

\item{col}{Named \code{character(4L)}. Colors for the labels, the character names
need to be "b", "y", "acxz" and "other", respectively for the b-ions,
y-ions, a,c,x,z-ions and the unidentified fragments.}

\item{labelCex}{\code{numeric(1)} giving the amount by which the text should be
magnified relative to the default. See parameter \code{cex} in \code{par()}.}

\item{labelSrt}{\code{numeric(1)} defining the rotation of the label.  See
parameter \code{srt} in \code{text()}.}

\item{labelAdj}{see parameter \code{adj} in \code{text()}.}

\item{labelPos}{see parameter \code{pos} in \code{text()}.}

\item{labelOffset}{see parameter \code{offset} in \code{text()}.}

\item{asp}{for \code{plotSpectraPTM()}, the target ratio (columns / rows) when
plotting mutliple spectra (e.g. for 20 spectra use asp = 4/5 for 4
columns and 5 rows or asp = 5/4 for 5 columns and 4 rows; see
\code{grDevices::n2mfrow()} for details).  If \code{deltaMz} is \code{TRUE}, \code{asp} is
ignored.}

\item{minorTicks}{\code{logical(1L)}. If \code{TRUE}, minor ticks are added to the
plots.  Default is set to \code{TRUE}.}

\item{USI}{\code{logical(1L)}. If \code{TRUE}, the universal spectrum identifier is
displayed.}

\item{...}{additional parameters to be passed to the \code{labelFragments()}
function.}
}
\value{
Creates a plot depicting an MS/MS-MS spectrum.
}
\description{
\code{plotSpectraPTM()} creates annotated visualisations of MS/MS spectra,
designed to explore fragment identifications and post-translational
modifications (PTMs).

\code{plotSpectraPTM()} plots a spectrum's m/z values on the x-axis and
corresponding intensities on the y-axis, labeling the peaks according to
theoretical fragment ions (e.g., b, y, a, c, x, z) computed using
\code{labelFragments()} and \code{calculateFragments()}.
}
\examples{
library("Spectra")

sp <- DataFrame(msLevel = 2L, rtime = 2345, sequence = "SIGFEGDSIGR")
sp$mz <- list(c(75.048614501953, 81.069030761719, 86.085876464844,
                88.039, 158.089569091797, 163.114898681641,
                173.128, 177.098587036133, 214.127075195312,
                232.137542724609, 233.140335083008, 259.938415527344,
                260.084167480469, 277.111572265625, 282.680786132812,
                284.079437255859, 291.208282470703, 315.422576904297,
                317.22509765625, 327.2060546875, 362.211944580078,
                402.235290527344, 433.255004882812, 534.258783,
                549.305236816406, 593.217041015625, 594.595092773438,
                609.848327636719, 631.819702148438, 632.324035644531,
                632.804931640625, 640.8193359375, 641.309936523438,
                641.82568359375, 678.357238769531, 679.346252441406,
                706.309623, 735.358947753906, 851.384033203125,
                880.414001464844, 881.40185546875, 906.396433105469,
                938.445861816406, 1022.56658935547, 1050.50415039062,
                1059.82800292969, 1107.52734375, 1138.521484375,
                1147.51538085938, 1226.056640625))
sp$intensity <- list(c(83143.03, 65473.8, 192735.53, 3649178.5,
                       379537.81, 89117.58, 922802.69, 61190.44,
                       281353.22, 2984798.75, 111935.03, 42512.57,
                       117443.59, 60773.67, 39108.15, 55350.43,
                       209952.97, 37001.18, 439515.53, 139584.47,
                       46842.71, 1015457.44, 419382.31, 63378.77,
                       444406.66, 58426.91, 46007.71, 58711.72,
                       80675.59, 312799.97, 134451.72, 151969.72,
                       1961975, 69405.76, 395735.62, 71002.98,
                       3215457.75, 136619.47, 166158.69, 682329.75,
                       239964.69, 242025.44, 1338597.62, 50118.02,
                       1708093.12, 43119.03, 97048.02, 2668231.75,
                       83310.2, 40705.72))
sp <- Spectra(sp)

## Annotate the spectum with the fragment labels
plotSpectraPTM(sp, main = "An example of an annotated plot")

## Annotate the spectrum without the delta m/z plot
plotSpectraPTM(sp, deltaMz = FALSE)

## Annotate the spectrum with different ion types
plotSpectraPTM(sp, type = c("a", "b", "x", "y"))

## Annotate the spectrum with variable modifications
plotSpectraPTM(sp, variable_modifications = c(R = 49.469))

## Annotate multiple spectra at a time
plotSpectraPTM(c(sp,sp), variable_modifications = c(R = 49.469))

## Color the peaks with different colors
plotSpectraPTM(sp, col = c(y = "red", b = "blue", acxy = "chartreuse3", other = "black"))
}
\seealso{
\code{\link[Spectra:spectra-plotting]{Spectra::plotSpectra()}}
}
\author{
Johannes Rainer, Sebastian Gibb, Guillaume Deflandre, Laurent Gatto
}
