#' @title Clustering of temporal patterns (Main function).
#'
#' @description The function performs a soft clustering of temporal patterns
#' based on the fuzzy c-means algorithm using the R package \code{Mfuzz}.
#'
#' @details All results are built from the results of our function
#' [DATAnormalization()].
#'
#' The \code{Mfuzz} package works with datasets where rows correspond to genes
#' and columns correspond to times.
#' If \code{RawCounts} (input of our function
#' [DATAprepSE()])
#' contains several replicates per time,
#' the algorithm computes the mean of replicates for each gene before using
#' [Mfuzz::mfuzz()].
#' When there are several biological conditions, the algorithm realizes the
#' [Mfuzz::mfuzz()]
#' analysis for each biological condition.
#'
#' @param SEresNorm Results of the function
#' [DATAnormalization()].
#' @param DATAnorm \code{TRUE} or \code{FALSE}. \code{TRUE} as default.
#' \code{TRUE} means the function uses the normalized data.
#' \code{FALSE} means the function uses the raw counts data.
#' @param DataNumberCluster Data.frame or \code{NULL}. \code{NULL} as default.
#' If \code{DataNumberCluster} is a data.frame where the first column contains
#' the name of the biological conditions and the second the number of cluster
#' selected for each biological condition.
#' If \code{DataNumberCluster=NULL}, a number of clusters will be automatically
#' computed for each biological condition
#' (see [MFUZZclustersNumber()]).
#' @param Method "kmeans" or "hcpc". The method used for selecting the number
#' of cluster to be used for the temporal cluster analysis
#' (see \code{Details}).
#' Only used if \code{DataNumberCluster} is not \code{NULL}.
#' @param Max.clust Integer strictly superior to 1 indicating the maximum
#' number of clusters.
#' \code{Max.clust} will be used only if \code{DataNumberCluster=NULL}
#' @param Membership Numeric value between 0 and 1.
#' For each cluster, genes with membership values below the threshold
#' \code{Membership} will not be displayed.
#' The membership values correspond to the probability of gene to belong
#' to each cluster.
#' @param Min.std Numeric positive value.
#' All genes where their standard deviations are smaller than the threshold
#' \code{Min.std} will be excluded.
#' @param Plot.Mfuzz \code{TRUE} or \code{FALSE}. \code{TRUE} as default.
#' If \code{TRUE}, all graphs will be plotted.
#' Otherwise no graph will be plotted.
#' @param path.result Character or \code{NULL}. Path to save all results.
#' If \code{path.result} contains a sub folder entitled
#' "1_UnsupervisedAnalysis_\code{Name.folder.mfuzz}" and a sub sub folder,
#' "1-4_MFUZZanalysis_\code{Name.folder.mfuzz}"
#' all results will be saved in the sub folder
#' "1_UnsupervisedAnalysis_\code{Name.folder.mfuzz}/
#' 1-4_MFUZZanalysis_\code{Name.folder.mfuzz}.
#' Otherwise, a sub folder entitled
#' "1_UnsupervisedAnalysis_\code{Name.folder.mfuzz}" and/or a sub sub folder
#' "1-4_MFUZZanalysis_\code{Name.folder.mfuzz}"
#' will be created in \code{path.result} and all results will be saved in
#' "1_UnsupervisedAnalysis_\code{Name.folder.mfuzz}/
#' 1-4_MFUZZanalysis_\code{Name.folder.mfuzz}".
#' If \code{NULL}, the results will not be saved in a folder.
#' \code{NULL} as default.
#' @param Name.folder.mfuzz Character or \code{NULL}.
#' If \code{Name.folder.mfuzz} is a character, the folder and sub folder names
#' which will contain the PCA graphs will respectively be
#' "1_UnsupervisedAnalysis_\code{Name.folder.mfuzz}" and
#' "1-4_MFUZZanalysis_\code{Name.folder.mfuzz}".
#' Otherwise, the folder and sub folder names will respectively be
#' "1_UnsupervisedAnalysis" and "1-4_MFUZZanalysis".
#'
#' @return The function returns the same SummarizedExperiment class object
#' \code{SEresNorm} with the different elements below
#' (saved in the metadata \code{Results[[1]][[4]]} of \code{SEresNorm})
#' * the final data used for the \code{Mfuzz} analysis (see \code{Details}).
#' * the cluster associated to each gene.
#' * plots generated by
#' [MFUZZclustersNumber()] and
#' [Mfuzz::mfuzz.plot2()]
#' for each biological condition.
#'
#' @seealso The function uses the function
#' [MFUZZclustersNumber()]
#' to compute the optimal number of cluster for each biological condition
#' with the kmeans method.
#'
#' @importFrom Mfuzz filter.NA fill.NA filter.std standardise mestimate
#' mfuzz mfuzz.plot2
#' @importFrom SummarizedExperiment colData assays rownames
#' @importFrom S4Vectors metadata
#' @importFrom Biobase assayData ExpressionSet
#' @importFrom ggplotify as.ggplot
#' @importFrom utils write.table
#' @importFrom grDevices dev.cur pdf dev.control dev.off dev.set recordPlot
#' @importFrom graphics par plot.new
#'
#' @export
#'
#' @examples
#' ## Data simulation
#' set.seed(33)
#' DATAclustSIM <- matrix(rnorm(12*10*3, sd=0.2,
#'                              mean=rep(c(rep(c(1, 6, 9, 4, 3, 1,
#'                                               6.5, 0.7, 10), times=2),
#'                                         rep(c(2, 3.6, 3.7, 5, 7.9, 8,
#'                                               7.5, 3.5, 3.4), times=2)),
#'                                       each=10)),
#'                        nrow=30, ncol=12)
#' DATAclustSIM <- floor(DATAclustSIM*100)
#' ##
#' colnames(DATAclustSIM) <- c("G1_t0_r1", "G1_t1_r1", "G1_t2_r1",
#'                             "G1_t0_r2", "G1_t1_r2", "G1_t2_r2",
#'                             "G2_t0_r3", "G2_t1_r3", "G2_t2_r3",
#'                             "G2_t0_r4", "G2_t1_r4", "G2_t2_r4")
#' ##------------------------------------------------------------------------##
#' ## Plot the temporal expression of each individual
#' graphics::matplot(t(rbind(DATAclustSIM[, 1:3], DATAclustSIM[, 4:6],
#'                           DATAclustSIM[, 7:9], DATAclustSIM[, 10:12])),
#'                   col=rep(c("black", "red"), each=6*10),
#'                   xlab="Time", ylab="Gene expression", type=c("b"), pch=19)
#'
#' ##------------------------------------------------------------------------##
#' ## Preprocessing step
#' DATAclustSIM <- data.frame(DATAclustSIM)
#'
#' resDATAprepSE <- DATAprepSE(RawCounts=DATAclustSIM,
#'                             Column.gene=NULL,
#'                             Group.position=1,
#'                             Time.position=2,
#'                             Individual.position=3)
#' ## Normalization
#' resNorm <- DATAnormalization(SEres=resDATAprepSE,
#'                              Normalization="rle",
#'                              Plot.Boxplot=FALSE,
#'                              Colored.By.Factors=FALSE)
#' ##------------------------------------------------------------------------##
#' resMFUZZ <- MFUZZanalysis(SEresNorm=resNorm,
#'                           DATAnorm=TRUE,
#'                           DataNumberCluster=NULL,
#'                           Membership=0.5,
#'                           Min.std=0.1,
#'                           Plot.Mfuzz=TRUE,
#'                           path.result=NULL)

MFUZZanalysis <- function(SEresNorm,
                          DATAnorm=TRUE,
                          DataNumberCluster=NULL,
                          Method="hcpc",
                          Max.clust=6,
                          Membership=0.5,
                          Min.std=0.1,
                          Plot.Mfuzz=TRUE,
                          path.result=NULL,
                          Name.folder.mfuzz=NULL) {
    ##-----------------------------------------------------------------------##
    ##-----------------------------------------------------------------------##
    ## Check
    resErr <- ErrMFUZZanalysis(SEresNorm=SEresNorm,
                               DATAnorm=DATAnorm,
                               DataNumberCluster=DataNumberCluster,
                               Method=Method,
                               Max.clust=Max.clust,
                               Membership=Membership,
                               Min.std=Min.std,
                               Plot.Mfuzz=Plot.Mfuzz,
                               path.result=path.result,
                               Name.folder.mfuzz=Name.folder.mfuzz)

    ##-----------------------------------------------------------------------##
    ##-----------------------------------------------------------------------##
    ## Folder creation if no existence
    if (is.null(Name.folder.mfuzz)) {
        Name.folder.mfuzz <- ""
        SubFolder.name <- "1_UnsupervisedAnalysis"
    } else {
        Name.folder.mfuzz <- paste0("_", Name.folder.mfuzz)
        SubFolder.name <- paste0("1_UnsupervisedAnalysis", Name.folder.mfuzz)
    }## if (is.null(Name.folder.mfuzz))

    SubFolderMFUZZ <- paste0("1-4_MFUZZanalysis", Name.folder.mfuzz)

    if (!is.null(path.result)) {
        if (!SubFolder.name%in%dir(path=path.result)) {
            print("Folder creation")
            dir.create(path=file.path(path.result, SubFolder.name))
        }## if(SubFolder.name%in%dir(path = path.result)==FALSE)
        path.result.f <- file.path(path.result, SubFolder.name)
    } else {
        path.result.f <- NULL
    }## if(is.null(path.result)==FALSE)

    if (!is.null(path.result.f)) {
        if(!SubFolderMFUZZ%in%dir(path=path.result.f)){
            dir.create(path=file.path(path.result.f, SubFolderMFUZZ))
        }## if(SubFolderMFUZZ%in%dir(path = path.result.f)==FALSE)
        path.result.new <- file.path(path.result.f, SubFolderMFUZZ)
    }else{
        path.result.new <- NULL
    }## if (!is.null(path.result.f))

    ##-----------------------------------------------------------------------##
    ##-----------------------------------------------------------------------##
    ## Preprocessing
    cSEdat <- SummarizedExperiment::colData(SEresNorm)

    if (c("Time")%in%colnames(cSEdat)) {
        Vect.time <- as.character(cSEdat$Time)
        LvlsTIME <- levels(as.factor(Vect.time))
        Nb.time <- length(LvlsTIME)
        Tps.info <- paste0("t", gsub("T", "", gsub("t", "", LvlsTIME)))
    } else {
        stop("Samples must belong to different times points.")
    }## if (c("Time")%in%colnames(cSEdat))

    if (c("Group")%in%colnames(cSEdat)) {
        Vect.group <- as.character(cSEdat$Group)
        Group.Lvls <- levels(as.factor(Vect.group))
        Nb.group <- length(Group.Lvls)

        Name.plot.Mfuzz <- c("ClustersNumbers",
                             paste0("Mfuzz.Plots.Group_", Group.Lvls))
        colname.grp <- paste0(".", rep(Group.Lvls, each=Nb.time))
    } else {
        Vect.group <- NULL
        Nb.group <- 1
        Group.Lvls <- "1condition"

        Name.plot.Mfuzz<-c("ClustersNumbers", "Mfuzz.Plots")
        colname.grp <- ""
    }## if (c("Group")%in%colnames(cSEdat))

    if (isTRUE(DATAnorm)) {
        aSE <- 2
    } else {
        aSE <- 1
    }## if (isTRUE(DATAnorm))

    ExprData <- SummarizedExperiment::assays(SEresNorm)[[aSE]]
    ExprData.f <- data.frame(ExprData)
    Name.G <- as.character(SummarizedExperiment::rownames(SEresNorm))

    BCplotsMfuzz <- vector(mode="list", length=Nb.group+1)
    names(BCplotsMfuzz) <- Name.plot.Mfuzz

    ##-----------------------------------------------------------------------##
    ## Selection the number of clusters
    if (is.null(DataNumberCluster)) {
        SEresClustNumber <- MFUZZclustersNumber(SEresNorm=SEresNorm,
                                                DATAnorm=DATAnorm,
                                                Method=Method,
                                                Max.clust=Max.clust,
                                                Min.std=Min.std,
                                                Plot.Cluster=Plot.Mfuzz,
                                                path.result=path.result.new)

        resCLnumber <- S4Vectors::metadata(SEresClustNumber)$Results[[1]][[4]]
        DataNumberCluster <- resCLnumber$DataClustSel
        Data.mfuzz <- resCLnumber$MfuzzData
        BCplotsMfuzz[[1]] <- resCLnumber$clustDATAplot

        fnidNBclusters <- TRUE

        if (isTRUE(Plot.Mfuzz)) {
            print(resCLnumber$clustDATAplot)
        }## if (isTRUE(Plot.Mfuzz))

    } else {

        Sum.clust <- sum(DataNumberCluster[, 2])
        fnidNBclusters <- FALSE

        ##-------------------------------------------------------------------##
        Data.mfuzz <- matrix(NA, ncol=Nb.time*Nb.group, nrow=nrow(ExprData.f))
        row.names(Data.mfuzz) <- Name.G ## row.names(ExprData)
        colnames(Data.mfuzz) <- paste0("Mean_", rep(Tps.info, times=Nb.group),
                                       colname.grp)

        ##-------------------------------------------------------------------##
        for (g in seq_len(Nb.group)) {
            for (t in seq_len(Nb.time)) {
                Index.t <- which(Vect.time == LvlsTIME[t])

                if (is.null(Vect.group)) {
                    ExprDataf_t <- as.data.frame(ExprData.f[, Index.t])
                    Data.mfuzz[, t] <- apply(ExprDataf_t, 1, mean)
                } else {
                    Index.g <- which(Vect.group == Group.Lvls[g])
                    Index.gt <- intersect(Index.t, Index.g)

                    ExprDataf.TG <- as.data.frame(ExprData.f[, Index.gt])
                    Data.mfuzz[, Nb.time*(g-1) + t] <- apply(ExprDataf.TG, 1,
                                                             FUN=mean)
                }## if (is.null(Vect.group))
            }## for (t in seq_len(Nb.time))
        }## for (g in seq_len(Nb.group))

    }## if (is.null(DataNumberCluster))

    ##-----------------------------------------------------------------------##
    ## data which will contain the Mfuzz results
    dat.mfuzz <- data.frame(matrix(NA, ncol=1+3*Nb.group, nrow=nrow(ExprData)))
    row.names(dat.mfuzz) <- row.names(Data.mfuzz)

    if (is.null(Vect.group)) {
        colnames(dat.mfuzz) <- c("Gene.Name", "Cluster", "Membership",
                                 paste0("Cluster.alpha_", Membership))
    } else {
        Pref.name <- c("Gene.Name",
                       rep(c("Cluster", "Membership",
                             paste0("Cluster.alpha_", Membership)),
                           times=Nb.group))
        Suff.name <- c("", rep(Group.Lvls, each=3))
        colnames(dat.mfuzz) <- paste0(Pref.name,
                                      c("", rep("_", times=3*Nb.group)),
                                      Suff.name)
    }## if (is.null(Vect.group))

    ##-----------------------------------------------------------------------##
    ## Mfuzz analysis
    Data.mfuzz.excl <- Data.mfuzz
    row.names(Data.mfuzz.excl) <- as.character(seq_len(nrow(Data.mfuzz)))

    for (g in seq_len(Nb.group)) {
        Smplg <- seq_len(Nb.time) + Nb.time*(g-1)
        ## eset <- methods::new('ExpressionSet', exprs=Data.mfuzz.excl[,Smplg])
        eset <- Biobase::ExpressionSet(assayData=Data.mfuzz.excl[,Smplg])
        eset.r <- Mfuzz::filter.NA(eset, thres=0.25)
        eset.f <- Mfuzz::fill.NA(eset.r, mode="mean")
        eset.fstd <- Mfuzz::filter.std(eset.f, min.std=Min.std, visu=FALSE)
        eset.s <- Mfuzz::standardise(eset.fstd)
        ## Estimation of the parameter m
        m.choose <- Mfuzz::mestimate(eset.s)
        ## c.choose <- cselection(eset, m.choose, crange=seq(4,32,4),
        ##                        repeats=5, visu=TRUE)
        ## cselection impossible when the data is too big

        if (Nb.group > 1) {
            Id.c.g <- which(DataNumberCluster[, 1] == Group.Lvls[g])
        } else {
            Id.c.g <- 1
        }## if(Nb.group>1)

        Nb.c.g <- DataNumberCluster[Id.c.g, 2]
        cl <- Mfuzz::mfuzz(eset.s, centers=Nb.c.g, m=m.choose)

        ##-------------------------------------------------------------------##
        ## Filling the data which will contain the Mfuzz results
        Cluster.alpha <- rep(0, times=length(cl$cluster))

        for (c in seq_len(length(cl$cluster))) {
            if (cl$membership[c, cl$cluster[c]] >= Membership) {
                Cluster.alpha[c] <- cl$cluster[c]
            }## if(cl$membership[c,cl$cluster[c]]>=Membership)
        }## for (c in seq_len(length(cl$cluster)))

        gene.not.excluded <- row.names(Biobase::assayData(eset.fstd)$exprs)
        gene.not.excluded <- as.numeric(gene.not.excluded)

        dat.mfuzz[gene.not.excluded, 3*(g-1)+1+1] <- as.numeric(cl$cluster)
        dat.mfuzz[gene.not.excluded, 3*(g-1)+2+1] <- round(apply(cl$membership,
                                                                 1, max),
                                                           digits=2)
        dat.mfuzz[gene.not.excluded, 3*(g-1)+3+1] <- Cluster.alpha

        ##-------------------------------------------------------------------##
        nrow.plot.f <- ceiling(sqrt(Nb.c.g))
        ncol.plot.f <- ceiling(Nb.c.g/ceiling(sqrt(Nb.c.g)))

        # resPlot <- MFUZZplot(eset=eset.s, cl=cl, min.mem=Membership,
        #                      NBcluster=Nb.c.g, time.labels=Tps.info,
        #                      MFUZZmfrow=c(nrow.plot.f, ncol.plot.f))

        # Re cemment commente
        resPlot <- ggplotify::as.ggplot(
            function() MFUZZplot(eset=eset.s, cl=cl, min.mem=Membership,
                                 NBcluster=Nb.c.g, time.labels=Tps.info,
                                 MFUZZmfrow=c(nrow.plot.f, ncol.plot.f))
        )

        BCplotsMfuzz[[g+1]] <- resPlot

        ##-------------------------------------------------------------------##
        ## Mfuzz plot
        if (is.null(Vect.group)) {
            Suff.save <- c("")
        } else {
            Suff.save <- paste0(Group.Lvls, "_")
        }## if (is.null(Vect.group))

        MfuzzTitle <- paste0("Mfuzz_TemporalClusters_",
                             Suff.save[g], "alpha", Membership, "_",
                             Name.folder.mfuzz, ".pdf")

        if (!is.null(path.result)) {
            grDevices::pdf(file=file.path(path.result.new, MfuzzTitle),
                           width=11, height=8)
            print(resPlot)
            grDevices::dev.off()
        }## if(is.null(path.result)==FALSE)

        if (isTRUE(Plot.Mfuzz)) {
            ## graphics::plot.new(), plot.new() set in MFUZZplot()
            print(resPlot)
        }## if (isTRUE(Plot.Mfuzz))

    }## for(g in seq_len(Nb.group))

    dat.mfuzz[, 1] <- Name.G

    if (isFALSE(fnidNBclusters)) {
        BCplotsMfuzz <- BCplotsMfuzz[-1]
    }## if (!is.null(DataNumberCluster))

    ##-----------------------------------------------------------------------##
    ##-----------------------------------------------------------------------##
    ## csv file with Mfuzz results
    CsvTitle <- paste0("Mfuzz_Results_", paste0(Group.Lvls, collapse="_"),
                       "_alpha", Membership, Name.folder.mfuzz, ".csv")

    if (!is.null(path.result)) {
        utils::write.table(dat.mfuzz, file=file.path(path.result.new, CsvTitle),
                           sep=";", row.names=FALSE)
    }## if(is.null(path.result)==FALSE)

    ##-----------------------------------------------------------------------##
    ##-----------------------------------------------------------------------##
    ## SE Mfuzz
    if (isTRUE(fnidNBclusters)) {## isFALSE(fnidNBclusters)
        SEresMFUZZ <- SEresClustNumber
        listMFUZZ <- append(list(Data.Mfuzz=as.data.frame(Data.mfuzz),
                                 Result.Mfuzz=dat.mfuzz),
                            BCplotsMfuzz)
    } else {
        SEresMFUZZ <- SEresNorm
        listMFUZZ <- append(list(DataClustSel=DataNumberCluster,
                                 Data.Mfuzz=as.data.frame(Data.mfuzz),
                                 Result.Mfuzz=dat.mfuzz),
                            BCplotsMfuzz)
    }## if (is.null(DataNumberCluster)) ## plotMfuzz=BCplotsMfuzz

    S4Vectors::metadata(SEresMFUZZ)$Results[[1]][[4]] <- listMFUZZ

    ##-----------------------------------------------------------------------##
    ##-----------------------------------------------------------------------##
    ## Output
    return(SEobj=SEresMFUZZ)
}## MFUZZanalysis()


MFUZZplot <- function(eset, cl, min.mem, time.labels, NBcluster, MFUZZmfrow) {
    ##-----------------------------------------------------------------------##
    ## ## clean up device : important to avoid problems when plotting all plots
    ## ## graphics::plot.new() ## put in main function
    ## cur_dev <- grDevices::dev.cur() ## store current device
    ## grDevices::pdf(file=NULL, width=11, height=8) ## open null device
    ## grDevices::dev.control("enable") ## turn on recording for the null device
    ## null_dev <- grDevices::dev.cur() ## store null device
    ## ## make sure we always clean up properly, even if smthg causes an error
    ## on.exit({
    ##     grDevices::dev.off(null_dev)
    ##     if (cur_dev > 1) grDevices::dev.set(cur_dev)
    ##     ## previous line: only set cur device if not null device
    ## })
    ##-----------------------------------------------------------------------##
    ## ylab="Expression changes"
    Mfuzz::mfuzz.plot2(eset=eset, cl=cl, min.mem=min.mem,
                       time.labels=time.labels, mfrow=MFUZZmfrow,
                       ylab="Standardised genes expression",
                       centre=TRUE, x11=FALSE)

    ##-----------------------------------------------------------------------##
    # The following lines ensure good ouput graphics
    if (NBcluster > prod(MFUZZmfrow)) {
        mod.gr <- NBcluster%%prod(MFUZZmfrow)
        for (p in seq_len(mod.gr)) {
            plot(0, type='n', axes=FALSE, ann=FALSE)
        }## for (p in seq_len(mod.gr))
    }## if(NBcluster > prod(MFUZZmfrow))

    if (NBcluster < prod(MFUZZmfrow)) {
        diff.gr <- prod(MFUZZmfrow) - NBcluster
        for (p in seq_len(diff.gr)) {
            plot(0, type='n', axes=FALSE, ann=FALSE)
        }## for(p in 1:diff.gr)
    }## if(NBcluster < prod(MFUZZmfrow))

    graphics::par(mfrow=c(1, 1))
    ##-----------------------------------------------------------------------##
    ## gMfuzz <- grDevices::recordPlot()
    ## return(gMfuzz)
}## MFUZZplot()

##---------------------------------------------------------------------------##
##---------------------------------------------------------------------------##
##---------------------------------------------------------------------------##
##---------------------------------------------------------------------------##

ErrMFUZZanalysis <- function(SEresNorm,
                             DATAnorm=TRUE,
                             DataNumberCluster=NULL,
                             Method="hcpc",
                             Max.clust=6,
                             Membership=0.5,
                             Min.std=0.1,
                             Plot.Mfuzz=TRUE,
                             path.result=NULL,
                             Name.folder.mfuzz=NULL) {
    ##-----------------------------------------------------------------------##
    ## Check 1
    if (!isTRUE(Plot.Mfuzz) & !isFALSE(Plot.Mfuzz)) {
        stop("'Plot.Mfuzz' must be TRUE or FALSE.")
    }## if (!isTRUE(Plot.Mfuzz) & !isFALSE(Plot.Mfuzz))

    if (!is.null(Name.folder.mfuzz)) {
        if (!is.character(Name.folder.mfuzz)) {
            stop("'Name.folder.mfuzz' must be NULL or a character.")
        }## if (!is.character(Name.folder.mfuzz))
    }## if (!is.null(Name.folder.mfuzz))

    if (!is.null(DataNumberCluster)) {
        Sum.clust <- sum(DataNumberCluster[, 2])
        ErrNcl_BC <- paste("The number of clusters selected for each",
                           "biological condition  must be an integer greater",
                           "or equal to 2.")
        if (Sum.clust < 2 | floor(Sum.clust) != Sum.clust) {
            stop(ErrNcl_BC)
        }## if(Sum.clust<2 | floor(Sum.clust)!=Sum.clust)
    }## if (!is.null(DataNumberCluster))

    ##-----------------------------------------------------------------------##
    ## Check 2
    resErr1 <- ErrMFUZZclustersNumber(SEresNorm=SEresNorm,
                                      DATAnorm=DATAnorm,
                                      Method=Method,
                                      Max.clust=Max.clust,
                                      Min.std=Min.std,
                                      Plot.Cluster=Plot.Mfuzz,
                                      path.result=path.result)

    ##-----------------------------------------------------------------------##
    return(Message="No error")
}## ErrMFUZZanalysis()

