.getRangedData <- function(obj) {
    if (is(obj, "RaggedExperiment"))
        unlist(as(obj, "GRangesList"))
    else if (is(obj, "RangedSummarizedExperiment"))
        rowRanges(obj)
    else
        stop("<internal> Don't know how to get range data from 'obj'")
}

#' @title Convert MultiAssayExperiment to MAF class
#'
#' @description Take a `MultiAssayExperiment` object with specific mutation
#'     assays and convert these into a `maftools` representation. The names
#'     provided via `synAssay` and `nonSynAssay` must match exactly those
#'     assays in the `MultiAssayExperiment`.
#'
#' @param x A `MultiAssayExperiment` object
#'
#' @param synAssay `character(1)` The name of the `ExperimentList` element in
#'   the `MultiAssayExperiment` that identifies synonymous variant
#'   classifications (default `"maf_syn"`).
#'
#' @param nonSynAssay `character(1)` The name of the `ExperimentList` element in
#'   the `MultiAssayExperiment` that identifies non-synonymous variant
#'   classifications (default `"maf_nonSyn"`).
#'
#' @return A `MAF` class object
#'
#' @seealso `?maftools::MAF`
#'
#' @export
MultiAssayExperimentToMAF <-
    function(x, synAssay = "maf_syn", nonSynAssay = "maf_nonSyn")
{
    BiocBaseUtils::checkInstalled("maftools")

    ns <- nonSynAssay %in% names(x)
    sy <- synAssay %in% names(x)

    if (!all(ns, sy))
        stop("'synAssay' or 'nonSynAssay' assays not in the 'ExperimentList'")

    maftools::MAF(
        nonSyn = .getRangedData(x[[nonSynAssay]]),
        syn = .getRangedData(x[[synAssay]]),
        clinicalData = as.data.frame(colData(x))
    )
}

