#' Common sequences bar plot
#' 
#' Creates an UpSetR bar plot showing the number of intersecting sequences across 
#' multiple samples.  This function is useful when more than 3 samples are being 
#' compared.
#' 
#' @param productive.aa A list data frames of of productive amino acid sequences 
#' generated by LymphoSeq function productiveSeq where the aggregate parameter 
#' was set to "aminoAcid".
#' @param samples The names of two or more samples in the productive.aa 
#' list whose intersections will shown.
#' @param color.sample The name of a single sample in the productive.aa list whose 
#' sequences will be colored in all samples that they appear in.
#' @param color.intersection The names of two or more samples in the productive.aa 
#' list whose intersections will be colored.
#' @param color A character vector of a color name that will be used highlight a selected 
#' sample or multiple sample intersections.
#' @param labels A character vector indicating whether the number of 
#' intersecting sequences should be shown on the tops of the bars.  Options 
#' include "yes" or "no".
#' @return Returns an UpSetR bar plot showing the number of intersecting sequences 
#' across multiple samples.
#' @examples
#' file.path <- system.file("extdata", "TCRB_sequencing", package = "LymphoSeq")
#' 
#' file.list <- readImmunoSeq(path = file.path)
#' 
#' productive.aa <- productiveSeq(file.list = file.list, aggregate = "aminoAcid")
#' 
#' commonSeqsBar(productive.aa = productive.aa, samples = c("TRB_CD4_949", "TRB_CD8_949", 
#' "TRB_Unsorted_949", "TRB_Unsorted_1320"), color.sample = "TRB_CD8_949")
#' @export
#' @import UpSetR
commonSeqsBar = function(productive.aa, samples, color.sample = NULL , 
                         color.intersection = NULL, color = "#377eb8", labels = "no"){
    unique.seqs = uniqueSeqs(productive.aa = productive.aa)
    sequence.matrix = seqMatrix(productive.aa = productive.aa, sequences = unique.seqs$aminoAcid)
    aminoAcid = sequence.matrix$aminoAcid
    sequence.matrix = sequence.matrix[,-c(1:2)]
    sequence.matrix[sequence.matrix > 0] <- 1
    sequence.matrix = cbind(aminoAcid, sequence.matrix)
    
    if(!is.null(color.sample)){
        queryFunction = function(row, sequence) {
            data <- (row["aminoAcid"] %in% sequence)
        }
        UpSetR::upset(sequence.matrix,
              sets = samples,
              nsets = length(samples), 
              nintersects = NA,
              mainbar.y.label = "Number of intersecting sequences",
              sets.x.label = "Number of sequences",
              line.size = 0.5,
              point.size = 1,
              text.scale = 1, 
              mb.ratio = c(0.7, 0.3),
              show.numbers = labels,
              matrix.dot.alpha = 0,
              query.legend = "bottom",
              queries = list(list(query = queryFunction, 
                                  params = list(productive.aa[[color.sample]]$aminoAcid), 
                                  color = "#377eb8",
                                  active = TRUE,
                                  query.name = color.sample)))
    }
    
    if(!is.null(color.intersection)){
        UpSetR::upset(sequence.matrix,
              sets = samples,
              nsets = length(samples), 
              nintersects = NA,
              mainbar.y.label = "Number of intersecting sequences",
              sets.x.label = "Number of sequences",
              line.size = 0.5,
              point.size = 1,
              text.scale = 1, 
              mb.ratio = c(0.7, 0.3),
              show.numbers = labels,
              matrix.dot.alpha = 0,
              queries = list(list(query = elements, 
                                  params = list(color.intersection), 
                                  color = color,
                                  active = TRUE)))
    }
    
    if(is.null(color.sample) & is.null(color.intersection)){
        UpSetR::upset(sequence.matrix,
              sets = samples,
              nsets = length(samples), 
              nintersects = NA,
              mainbar.y.label = "Number of intersecting sequences",
              sets.x.label = "Number of sequences",
              line.size = 0.5,
              point.size = 1,
              text.scale = 1, 
              mb.ratio = c(0.7, 0.3),
              show.numbers = labels,
              matrix.dot.alpha = 0)
    }
}
