% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit.R
\name{fitDOU}
\alias{fitDOU}
\title{Fit Differential ORF Usage Models}
\usage{
fitDOU(
  data,
  formula = ~condition * strategy,
  specs = ~condition * strategy,
  dispformula = NULL,
  dispersion_modeling = "auto",
  lrt = FALSE,
  diagnostic = FALSE,
  parallel = list(n = 4L, autopar = TRUE),
  optimizers = FALSE,
  verbose = TRUE
)
}
\arguments{
\item{data}{A \code{\link{DOUData-class}} object containing raw ORF-level
counts and sample metadata. This object is used as the input for
modeling DOU within the \pkg{DOTSeq} framework.}

\item{formula}{A formula object specifying the model design,
e.g., \code{~ condition * strategy}.}

\item{specs}{A formula specifying the structure of the estimated
marginal means. Default is \code{~condition * strategy}.}

\item{dispformula}{Optional formula object specifying a custom dispersion
model (used when \code{dispersion_modeling = "custom"}).}

\item{dispersion_modeling}{Character string specifying the dispersion
modeling strategy. Options are:
\describe{
\item{\code{"auto"}}{
Fit both strategy-dependent and shared dispersion models, and
select the best via likelihood ratio test.
}
\item{\code{"strategy"}}{
Model dispersion as a function of sequencing strategy.
}
\item{\code{"shared"}}{
Assume constant dispersion across all predictor levels.
}
\item{\code{"custom"}}{
Use a user-specified dispersion formula via \code{dispformula}.
}
}}

\item{lrt}{Logical; if \code{TRUE}, performs a likelihood ratio test to
compare the full model (with interaction) against a reduced model
(without interaction) to assess translation-specific effects.
Default is \code{FALSE}.}

\item{diagnostic}{Logical; if \code{TRUE}, runs \pkg{DHARMa}
diagnostics to assess model fit. Default is \code{FALSE}.}

\item{parallel}{A list passed to \code{glmmTMBControl} to configure
parallel optimization, e.g., \code{list(parallel = TRUE, ncpus = 4)}.
Default is \code{list(n = 4L, autopar = TRUE)}.}

\item{optimizers}{Logical; if \code{TRUE}, enables brute-force
optimization using multiple optimizers in
\code{\link[glmmTMB]{glmmTMBControl}}. Default is \code{FALSE}.}

\item{verbose}{Logical; if \code{TRUE}, prints progress messages.
Default is \code{TRUE}.}
}
\value{
A named \code{list} of \code{PostHoc} objects, one per ORF.
}
\description{
This function fits beta-binomial generalized (mixed) linear models
(GLM or GLMM) for Differential ORF Usage (DOU) across all genes
(via \code{\link[glmmTMB]{glmmTMB}}). It supports multiple dispersion
modeling approaches and optional diagnostics using \pkg{DHARMa}.
}
\examples{
# Load SummarizedExperiment to enable subsetting and access to components
# like rowRanges and rowData
library(SummarizedExperiment)

# Read in count matrix, condition table, and annotation files
dir <- system.file("extdata", package = "DOTSeq")

cnt <- read.table(
    file.path(dir, "featureCounts.cell_cycle_subset.txt.gz"),
    header = TRUE,
    comment.char = "#"
)
names(cnt) <- gsub(".*(SRR[0-9]+).*", "\\\\1", names(cnt))

gtf <- file.path(dir, "gencode.v47.orf_flattened_subset.gtf.gz")
bed <- file.path(dir, "gencode.v47.orf_flattened_subset.bed.gz")

meta <- read.table(file.path(dir, "metadata.txt.gz"))
names(meta) <- c("run", "strategy", "replicate", "treatment", "condition")
cond <- meta[meta$treatment == "chx", ]
cond$treatment <- NULL

# Create a DOTSeqDataSets object
d <- DOTSeqDataSetsFromFeatureCounts(
    count_table = cnt,
    condition_table = cond,
    flattened_gtf = gtf,
    flattened_bed = bed
)

dou <- getDOU(d)

# Keep ORFs where all replicates in at least one condition pass min_count
# Single-ORF genes are removed
dou <- dou[rowRanges(dou)$is_kept == TRUE, ]

# Randomly sample 100 ORFs for fitDOU
set.seed(42)
random_rows <- sample(seq_len(nrow(dou)), size = 100)
dou <- dou[random_rows, ]

# Model fitting using fitDOU
rowData(dou)[["DOUResults"]] <- fitDOU(
    data = dou,
    formula = ~ condition * strategy,
    specs = ~ condition * strategy,
    dispersion_modeling = "auto",
    lrt = FALSE,
    optimizers = FALSE,
    diagnostic = FALSE,
    parallel = list(n = 4L, autopar = TRUE),
    verbose = TRUE
)

}
\references{
Brooks, M. E., Kristensen, K., van Benthem, K. J., Magnusson, A.,
Berg, C. W., Nielsen, A., Skaug, H. J., Mächler, M. and Bolker, B. M.
(2017). glmmTMB balances speed and flexibility among packages for
zero-inflated generalized linear mixed modeling. The R Journal, 378–400.
DOI: 10.32614/RJ-2017-066

Lenth R, Piaskowski J (2025). emmeans: Estimated Marginal Means, aka
Least-Squares Means. R package version 2.0.0.
\url{https://rvlenth.github.io/emmeans/}

Hartig F (2025). DHARMa: Residual Diagnostics for Hierarchical
(Multi-Level / Mixed) Regression Models. R package version 0.4.7.
\url{https://github.com/florianhartig/dharma}
}
\seealso{
\code{\link{DOTSeq}}, \code{\link{DOTSeqDataSets-class}},
\code{\link{testDOU}}
}
