% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/DOTSeq.R, R/main.R
\docType{package}
\name{DOTSeq}
\alias{DOTSeq-package}
\alias{DOTSeq}
\title{DOTSeq: Differential Analysis of Translation Efficiency and Usage of Open Reading Frames}
\usage{
DOTSeq(
  datasets,
  formula = ~condition * strategy,
  modules = c("DOU", "DTE"),
  target = NULL,
  baseline = NULL,
  min_count = 1,
  stringent = TRUE,
  dispersion_modeling = "auto",
  dispformula = NULL,
  lrt = FALSE,
  diagnostic = FALSE,
  parallel = list(n = 4L, autopar = TRUE),
  optimizers = FALSE,
  nullweight = 500,
  contrasts_method = "revpairwise",
  verbose = TRUE
)
}
\arguments{
\item{datasets}{Either:
\describe{
\item{\code{\link{DOTSeqDataSets-class}} object}{
A pre-constructed \code{\link{DOTSeqDataSets-class}} object created using
\code{\link{DOTSeqDataSets-class}}. It must include:
\describe{
\item{\code{DOU}}{
A \code{\link{DOUData-class}} object containing filtered
raw counts, sample metadata, and ORF-level annotations.
}
\item{\code{DTE}}{
A \code{\link[DESeq2]{DESeqDataSet-class}} object used for
modeling DTE via DESeq2.
}
}
If a \code{\link{DOTSeqDataSets-class}} object is provided, the function skips raw input
parsing and uses these objects directly.
}
}}

\item{formula}{A formula object specifying the design.
Default is \code{~ condition * strategy}.}

\item{modules}{Character vector specifying which \pkg{DOTSeq} modules
to run. Options include \code{"DOU"} and \code{"DTE"}.
Default is \code{c("DOU", "DTE")}.}

\item{target}{Character string specifying the non-reference condition
level to extract the corresponding interaction term.
Default is \code{NULL}.}

\item{baseline}{Character string specifying the desired reference level.
Default is \code{NULL}.}

\item{min_count}{Minimum count threshold for filtering ORFs.
Default is \code{1}.}

\item{stringent}{Logical or \code{NULL}; determines the filtering strategy:
\describe{
\item{\code{TRUE}}{
Keep ORFs where all replicates in at least one condition pass
\code{min_count}.
}
\item{\code{FALSE}}{
Keep ORFs where all replicates in at least one condition-strategy
group pass \code{min_count}.
}
\item{\code{NULL}}{
Keep ORFs where total counts across replicates pass
\code{min_count}.
}
}}

\item{dispersion_modeling}{String specifying the dispersion modeling
approach for DOU. Options include \code{"auto"}, \code{"shared"},
or \code{"custom"}. Default is \code{"auto"}.}

\item{dispformula}{Optional formula object for custom dispersion modeling.}

\item{lrt}{Logical; if \code{TRUE}, performs a likelihood ratio test
comparing full vs reduced models in DOU.
Default is \code{FALSE}.}

\item{diagnostic}{Logical; if \code{TRUE}, enables model diagnostics in DOU.
Default is \code{FALSE}.}

\item{parallel}{A list passed to \code{\link[glmmTMB]{glmmTMBControl}} to configure parallel
optimization in DOU. Default is \code{list(n = 4L, autopar = TRUE)}.}

\item{optimizers}{Logical; if \code{TRUE}, enables brute-force optimization
using multiple optimizers in \code{\link[glmmTMB]{glmmTMBControl}}.
Default is \code{FALSE}.}

\item{nullweight}{Numeric. Prior weight on the null hypothesis for empirical
Bayes shrinkage in DOU. Default is \code{500}.}

\item{contrasts_method}{Character string specifying the method for post hoc
contrasts in DOU. Default is \code{"revpairwise"}.}

\item{verbose}{Logical; if \code{TRUE}, prints progress messages.
Default is \code{TRUE}.}
}
\value{
A \code{\link{DOTSeqDataSets-class}} object containing:
\describe{
\item{\code{DOU}}{
A \code{\link{DOUData-class}} object with DOU results.
}
\item{\code{DTE}}{
A \code{\link{DTEData-class}} object with DTE results.
}
}
}
\description{
DOTSeq provides a flexible beta-binomial generalized linear model (GLM)
framework for modeling the expected proportion of ribosome profiling
(Ribo-seq) to RNA-seq counts for individual open reading frames (ORFs)
relative to other ORFs within the same gene. It also includes a negative
binomial GLM framework for detecting changes in translation efficiency
across experimental conditions.

\pkg{DOTSeq} is a statistical framework for modeling Differential ORF
Translation using ribosome profiling (Ribo-seq) and RNA-seq data. It
supports two modes of input:
\itemize{
\item A named list of raw input components: \code{count_table},
\code{condition_table}, \code{flattened_gtf}, and \code{bed}.
\item A pre-constructed \code{\link{DOTSeqDataSets-class}} object containing processed data.
}

The function automatically detects the input type and proceeds with the
appropriate workflow. It performs ORF-level filtering, model fitting,
post hoc contrasts, and adaptive shrinkage of effect sizes. Plotting and
downstream analysis are handled separately via the \code{\link{plotDOT}} function.
}
\examples{
# Load test data
dir <- system.file("extdata", package = "DOTSeq")

cnt <- read.table(
    file.path(dir, "featureCounts.cell_cycle_subset.txt.gz"),
    header = TRUE,
    comment.char = "#"
)
names(cnt) <- gsub(".*(SRR[0-9]+).*", "\\\\1", names(cnt))

gtf <- file.path(dir, "gencode.v47.orf_flattened_subset.gtf.gz")
bed <- file.path(dir, "gencode.v47.orf_flattened_subset.bed.gz")

meta <- read.table(file.path(dir, "metadata.txt.gz"))
names(meta) <- c("run", "strategy", "replicate", "treatment", "condition")
cond <- meta[meta$treatment == "chx", ]
cond$treatment <- NULL

# Use raw input list
d <- DOTSeqDataSetsFromFeatureCounts(
    count_table = cnt,
    condition_table = cond,
    flattened_gtf = gtf,
    flattened_bed = bed
)

d <- DOTSeq(datasets = d, modules = "DTE")

show(d)

# Get the DOUData object
dou <- getDOU(d)

# Subset DOUData and edit DOTSeqDataSets in place
set.seed(42)
random_rows <- sample(seq_len(nrow(dou)), size = 50)
getDOU(d) <- dou[random_rows, ]

# Run the DOU module
d <- DOTSeq(datasets = d)

# Create a DOTSeqDataSets object and use it as input
d <- DOTSeqDataSetsFromFeatureCounts(
    count_table = cnt,
    condition_table = cond,
    flattened_gtf = gtf,
    flattened_bed = bed
)
d <- DOTSeq(datasets = d, modules = "DTE")

}
\references{
Brooks, M. E., Kristensen, K., van Benthem, K. J., Magnusson, A., Berg,
C. W., Nielsen, A., Skaug, H. J., Mächler, M. and Bolker, B. M. (2017).
glmmTMB balances speed and flexibility among packages for zero-inflated
generalized linear mixed modeling. The R Journal, 378–400.
DOI: 10.32614/RJ-2017-066

Love, M. I., Huber, W., Anders, S. (2014).
Moderated estimation of fold change and dispersion for RNA-seq data with
DESeq2. Genome Biology, 15:550. DOI: 10.1186/s13059-014-0550-8

Lenth, R., Piaskowski, J. (2025).
emmeans: Estimated Marginal Means, aka Least-Squares Means.
R package version 2.0.0. \url{https://rvlenth.github.io/emmeans/}

Stephens, M. (2016).
False discovery rates: a new deal. Biostatistics, 18(2).
DOI: 10.1093/biostatistics/kxw041

Hartig, F. (2025).
DHARMa: Residual Diagnostics for Hierarchical (Multi-Level / Mixed)
Regression Models. R package version 0.4.7.
\url{https://github.com/florianhartig/dharma}
}
\seealso{
Useful links:
\itemize{
  \item \url{https://github.com/compgenom/DOTSeq}
  \item Report bugs at \url{https://github.com/compgenom/DOTSeq/issues}
}


\code{\link{DOTSeqDataSets-class}}, \code{\link{fitDOU}},
\code{\link{testDOU}}, \code{\link{plotDOT}}
}
\author{
\strong{Maintainer}: Chun Shen Lim \email{lim.bioinfo@gmail.com} (\href{https://orcid.org/0000-0001-7015-0125}{ORCID})

Authors:
\itemize{
  \item Gabrielle Chieng [contributor]
}

Other contributors:
\itemize{
  \item Marsden [funder]
}

}
