% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/primary.R
\name{stabilitySelection}
\alias{stabilitySelection}
\title{Stability selection calculates selection probabilities for every
possible feature-feature interaction within the input data}
\usage{
stabilitySelection(
  object,
  subSample = FALSE,
  nreps = 500,
  optimal_lambda,
  assay,
  BPPARAM = bpparam(),
  BPOPTIONS = bpoptions()
)
}
\arguments{
\item{object}{A \code{\link[=DNEA-class]{DNEA}} object.}

\item{subSample}{TRUE/FALSE indicating whether the number of samples
are unevenly split by condition and subsampling should be performed
when randomly sampling to even out the groups.}

\item{nreps}{The total number of replicates to perform in stability
selection. The default is 1000.}

\item{optimal_lambda}{\emph{\strong{OPTIONAL -}} The optimal lambda value to be
used in the model. This parameter is only necessary if
\code{\link{BICtune}} is not performed prior.}

\item{assay}{A character string indicating which expression assay to
use for analysis. The default is the "log_scaled_data" assay that is
created during \code{\link{createDNEAobject}}.}

\item{BPPARAM}{a BiocParallel object.}

\item{BPOPTIONS}{a list of options for BiocParallel created using
the \code{\link[BiocParallel:bpoptions]{bpoptions}} function.}
}
\value{
A \code{\link[=DNEA-class]{DNEA}} object after populating the
stable_networks slot of the object. It contains the selection
results from stability selection as well as the calculated
selection probabilities.
}
\description{
This function randomly samples the input data and fits a glasso model
with the sampled data for \strong{\emph{nreps}} number of replicates. The
resulting adjacency matrices are summed together and selection probabilities
for each feature-feature interaction are calculated. Stability selection is
particularly useful for smaller data sets. A large number of replicates
should be performed (the default is 1000). The exact method deployed
varies slightly whether or not additional sub-sampling of the data is
performed. More information can be found in the
\strong{\emph{Details}} section.
}
\details{
Stability selection provides an additional approach by which to
regularize the network model and create more robust results, particularly
when \strong{\emph{p >> n}}. Stability selection works by randomly
sampling (without replacement) the input data many times and fitting
a glasso model to each subset of sampled data. The unwieghted adjacency
matrix from each model is summed together (A feature-feature interaction
is considered present if the partial correlation value is above 1e-5),
and the probability of an edge being selected in a random subset of the
data is calculated by dividing the number of times an edge was selected in
the replicates over the total number of replicates. This results in a
selection probability for every possible feature-feature interaction
that is used to modify the regularization parameter
via the following equation:
\deqn{\rho = \lambda*(1 / (0.0001 + selection.probability))}{ rho = lambda*(1 / (0.0001 + selection.probability))}

However, when the sample groups are very unbalanced, randomly
sampling strongly favors the larger group, resulting in over
representation. In order to combat this, setting subSample=TRUE
modifies the random sample by sub-sampling the experimental groups
to even out the sample numbers. In this method, 90\% of the smaller
group is randomly sampled without replacement, and an
additional 10\% is randomly sampled without replacement from
the entire group to preserve the variance. The larger group
is randomly sampled to have 1.3 times the number of samples
present in the smaller group. This method ensures that each
group is equally represented in stability selection. \cr

The principles of stability selection remain similar with both methods,
however, there are a few small differences. Stability selection
\emph{without} additional sub-sampling randomly samples 50\% of each group
(without replacement) and fits a model for both halves of the sampled data.
Since nearly all of the data for the smaller group is used \emph{with}
additional sub-sampling, only one model is fit per replicate when
subSample=TRUE. This means that at the default value of nreps=500,
1000 randomly sampled models are fit in total \emph{without} sub-sampling,
but 500 randomly sampled models are fit in total \emph{with} sub-sampling.
More details about the stability approach deployed in this function can be
found in Ma et al. (2019) referenced below.
}
\examples{
#import BiocParallel package
library(BiocParallel)

#load example data
data(TEDDY)
data(T1Dmeta)

#make sure metadata and expression data are in same order
TEDDY <- TEDDY[seq(50), ]
T1Dmeta <- T1Dmeta[colnames(TEDDY),]

#create group labels
group_labels <- T1Dmeta$group
names(group_labels) <- rownames(T1Dmeta)

#initiate DNEA object
dnw <- createDNEAobject(project_name = "test", expression_data = TEDDY,
                            group_labels = group_labels)

#optimize lambda parameter
dnw <- BICtune(object=dnw,
               informed=TRUE,
               interval=0.01)

# perform stability selection
dnw <- stabilitySelection(object=dnw,
                          subSample=FALSE,
                          nreps=4,
                          BPPARAM=bpparam())

}
\references{
Ma J, Karnovsky A, Afshinnia F, Wigginton J, Rader DJ,
Natarajan L, Sharma K, Porter AC, Rahman M, He J, Hamm L, Shafi T,
Gipson D, Gadegbeku C, Feldman H, Michailidis G, Pennathur S.
Differential network enrichment analysis reveals novel lipid
pathways in chronic kidney disease. Bioinformatics.
2019 Sep 15;35(18):3441-3452. doi: 10.1093/bioinformatics/btz114.
PMID: 30887029; PMCID: PMC6748777.
\url{https://www.ncbi.nlm.nih.gov/pmc/articles/PMC6748777/}

Nicolai, M., & Peter, B. (2010). Stability selection.
Journal of the Royal Statistical Society: Series B
(Statistical Methodology), 72(4), 417-473.
\url{https://stat.ethz.ch/Manuscripts/buhlmann/stability.pdf}
}
\seealso{
\code{\link{selectionProbabilities}},
\code{\link{selectionResults}},
\code{\link[BiocParallel:bpparam]{bpparam}},
\code{\link[BiocParallel:bpoptions]{bpoptions}}
\code{\link[glasso:glasso]{glasso}}
}
\author{
Christopher Patsalis
}
