% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ChromBackendSpectra.R
\name{ChromBackendSpectra}
\alias{ChromBackendSpectra}
\alias{backendInitialize,ChromBackendSpectra-method}
\alias{chromSpectraIndex}
\title{Chromatographic Data Backend for Spectra Objects}
\usage{
ChromBackendSpectra()

\S4method{backendInitialize}{ChromBackendSpectra}(
  object,
  spectra = Spectra(),
  factorize.by = c("msLevel", "dataOrigin"),
  summarize.method = c("sum", "max"),
  chromData = fillCoreChromVariables(),
  spectraVariables = character(),
  ...
)

chromSpectraIndex(object)
}
\arguments{
\item{object}{A \code{ChromBackendSpectra} object.}

\item{spectra}{A \code{Spectra} object.}

\item{factorize.by}{A \code{character} vector of \code{spectraVariables} for grouping
\code{Spectra} data into chromatographic data (i.e., creating separate
chromatograms for each unique combination of these variables).
Default: \code{c("msLevel", "dataOrigin")}, which creates one chromatogram
per MS level per data file.
If \code{chromData} is provided, it \strong{must} also contain these columns.}

\item{summarize.method}{A \code{character} string specifying intensity summary:
\code{"sum"} (default) or \code{"max"}.}

\item{chromData}{A \code{data.frame} with chromatographic data for use in
\code{backendInitialize()}. If missing, a default is generated. Columns
like \code{rtMin}, \code{rtMax}, \code{mzMin}, and \code{mzMax} must be provided and not
contain \code{NA} values. Use \code{-Inf/Inf} for unspecified values. The
\code{"dataOrigin"} column must match the \code{Spectra} object's
\code{"dataOrigin"}.}

\item{spectraVariables}{A \code{character} vector specifying which variables
from the \code{Spectra} object should be added to the chromData. These
will be mapped using the \code{chromSpectraIndex} variable.}

\item{...}{Additional parameters.}
}
\value{
Refer to the individual function description for information on the
return value.
}
\description{
The \code{ChromBackendSpectra} class extends \code{ChromBackendMemory}, inheriting
all its slots and methods while providing additional functionality for
summarizing chromatographic data from \code{\link[Spectra:Spectra]{Spectra::Spectra()}} objects.

It can be initialized with a \code{Spectra} object, which is stored in the
\code{spectra} slot of the backend. Users can also provide a \code{data.frame}
containing chromatographic metadata, stored in \verb{@chromData}. This metadata
filters the \code{Spectra} object and generates \code{peaksData}. If \code{chromData} is
not provided, a default \code{data.frame} is created from the \code{Spectra} data.
An "rtMin", "rtMax", "mzMin", and "mzMax" column will be created by
condensing the \code{Spectra} data corresponding to each unique combination of
the \code{factorize.by} variables.

By "factorization" we mean the process of grouping spectra
into chromatograms based on specified variables. For example, using
\code{factorize.by = c("msLevel", "dataOrigin")} means that all MS1 spectra from
file "A" form one chromatogram, all MS2 spectra from file "A" form another,
and so on. Each unique combination of the factorization variables creates
a separate chromatogram. This is essential for organizing spectral data into
meaningful chromatographic traces that can be visualized and analyzed.

The \emph{dataOrigin} core chromatogram variable should reflect the \emph{dataOrigin}
of the \code{Spectra} object. The \code{factorize.by} parameter defines the variables
for grouping \code{Spectra} data into chromatographic data. The default is
\code{c("msLevel", "dataOrigin")}, which will define separate chromatograms for
each combination of \code{msLevel} and \code{dataOrigin}. These variables must be in
both the \code{spectraData()} of the \code{Spectra} and \code{chromData} (if provided).

The \code{summarize.method} parameter defines how spectral data intensity is
summarized:
\itemize{
\item \strong{"sum"}: Sums intensity to create a Total Ion Chromatogram (TIC).
\item \strong{"max"}: Takes max intensity for a Base Peak Chromatogram (BPC).
}

If \code{chromData} or its factorization columns are modified, the \code{factorize()}
method must be called to update \code{chromSpectraIndex}.
}
\details{
No \code{peaksData} is stored until the user calls a function that generates it
(e.g., \code{rtime()}, \code{peaksData()}, \code{intensity()}). The \verb{@peaksData} slot
replacement is unsupported — modifications are temporary to optimize memory.
The \verb{@inMemory} slot indicates this with \code{TRUE}.

\strong{Spectra Sort Index}: The \code{ChromBackendSpectra} backend maintains a
\code{spectraSortIndex} slot that stores a sort order for the internal \code{Spectra}
object based on \code{dataOrigin} and \code{rtime}. To optimize performance, the sort
index is only computed and stored when the spectra are unsorted; if already
sorted (which is typical for most real-world data), \code{spectraSortIndex} remains
empty (\code{integer()}). This avoids unnecessary subsetting operations. The sort
index is automatically recalculated whenever the \code{factorize()} method is called,
ensuring it remains valid and consistent. This approach avoids the need to
physically reorder disk-backed \code{Spectra} objects, which would require loading
all data into memory.

\strong{Factorize and Subsetting}: The \code{factorize()} method updates the
\code{chromSpectraIndex} in both \code{chromData} and the \verb{@spectra} to reflect
the current grouping, and recalculates \code{spectraSortIndex} to maintain the
correct sort order. The \code{[} subsetting operator properly handles subsetting
of both \verb{@chromData}, \verb{@peaksData}, and \verb{@spectra}, while updating the
\code{spectraSortIndex} to reference valid positions in the subsetted data.

\code{ChromBackendSpectra} should reuse \code{ChromBackendMemory} methods whenever
possible to keep implementations simple.
}
\examples{
library(Spectra)
library(MsBackendMetaboLights)

## Get Spectra data from MetaboLights
be <- backendInitialize(MsBackendMetaboLights(),
    mtblsId = "MTBLS39",
    filePattern = c("63B.cdf")
)
s <- Spectra(be)

s <- setBackend(s, MsBackendMemory())

## Initialize ChromBackendSpectra
be_empty <- new("ChromBackendSpectra")
be <- backendInitialize(be_empty, s)

## replace the msLevel data
msLevel(be) <- c(1L, 2L, 3L)

## re-factorize the data
be <- factorize(be)

## Create BPC : we summarize the intensity present in the Spectra object
## by the maximum value, thus creating a Base Peak Chromatogram.
be <- backendInitialize(be_empty, s, summarize.method = "max")

## Can now see the details of this bpc by looking at the chromData of our
## object
chromData(be)

## Another possibilities is to create eics from the Spectra object.
## Here we create an EIC with a specific m/z and retention time window.
df <- data.frame(mzMin = 100.01, mzMax = 100.02 , rtMin = 50, rtMax = 100)
be <- backendInitialize(be_empty, s, summarize.method = "sum")
chromData(be) <- cbind(chromData(be), df)

## now when we call the peaksData function, we will get the intensity
## of the spectra object that are in the m/z and retention time window
## defined in the chromData.
peaksData(be)

}
\author{
Philippine Louail, Johannes Rainer.
}
