% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/utilities.R
\name{ClustersList}
\alias{ClustersList}
\alias{asClusterization}
\alias{toClustersList}
\alias{fromClustersList}
\alias{groupByClustersList}
\alias{groupByClusters}
\alias{mergeClusters}
\alias{multiMergeClusters}
\title{\emph{Clusters} utilities}
\usage{
asClusterization(clusters, allCells = NULL)

toClustersList(clusters)

fromClustersList(
  clustersList,
  elemNames = vector(mode = "character"),
  throwOnOverlappingClusters = TRUE
)

groupByClustersList(elemNames, clustersList, throwOnOverlappingClusters = TRUE)

groupByClusters(clusters)

mergeClusters(clusters, names, mergedName = "")

multiMergeClusters(clusters, namesList, mergedNames = NULL)
}
\arguments{
\item{clusters}{A named \code{vector} or \code{factor} that defines the \emph{clusters}}

\item{allCells}{A \code{vector} of cells' names that should list the same names in
the \code{clusters} in any order}

\item{clustersList}{A named \code{list} whose elements define the various clusters}

\item{elemNames}{A \code{list} of names to which associate a cluster}

\item{throwOnOverlappingClusters}{When \code{TRUE}, in case of overlapping
clusters, the function \code{fromClustersList} and \code{groupByClustersList} will
throw. This is the default. When FALSE, instead, in case of overlapping
clusters, \code{fromClustersList} will return the last cluster to which each
element belongs, while \code{groupByClustersList} will return a vector of
positions that is longer than the given \code{elemNames}}

\item{names}{A list of \emph{clusters} names to be merged}

\item{mergedName}{The name of the new merged clusters}

\item{namesList}{A \code{list} of \code{list}s of \emph{clusters} names to be respectively
merged}

\item{mergedNames}{The names of the new merged \emph{clusters}}
}
\value{
\code{asClusterization()} returns the \emph{clusterization} as a named
\code{factor}

\code{toClustersList()} returns a \code{list} of clusters

\code{fromClustersList()} returns a clusterization. If the given
\code{elemNames} contain values not present in the \code{clustersList}, those will be
marked as \code{"-1"}

\code{groupByClusters()} and \code{groupByClustersList()} return a permutation
that groups the clusters together. For each cluster the positions are
guaranteed to be in increasing order. In case, all elements not
corresponding to any cluster are grouped together as the last group

\code{mergeClusters()} returns a new \emph{clusterization} with the wanted
\emph{clusters} being merged. If less than 2 \emph{cluster} names were passed the
function will emit a warning and return the initial \emph{clusterization}

\code{multiMergeClusters()} returns a new \emph{clusterization} with the
wanted \emph{clusters} being merged by consecutive iterations of
\code{\link[=mergeClusters]{mergeClusters()}} on the given \code{namesList}
}
\description{
Handle \emph{clusterization} <-> \emph{clusters} \code{list} conversions,
\emph{clusters} grouping and merge
}
\details{
\code{asClusterization()} given a \emph{clusterization} in the form of a
\code{data.frame} or a \code{vector} or a \code{factor}, returns a named \code{factor}

\code{toClustersList()} given a \emph{clusterization}, creates a \code{list} of
\emph{clusters} (i.e. for each \emph{cluster}, which elements compose the \emph{cluster})

\code{fromClustersList()} given a \code{list} of \emph{clusters} returns a
\emph{clusterization} (i.e. a named \code{vector} that for each element indicates to
which cluster it belongs)

\code{groupByClusters()} given a \emph{clusterization} returns a permutation,
such that using the permutation on the input the \emph{clusters} are grouped
together

\code{groupByClustersList()} given the elements' names and a \code{list} of
\emph{clusters} returns a permutation, such that using the permutation on the
given names the \emph{clusters} are grouped together.

\code{mergeClusters()} given a \emph{clusterization}, creates a new one where
the given \emph{clusters} are merged.

\code{multiMergeClusters()} given a \emph{clusterization}, creates a new one
where the given sets of \emph{clusters} are merged.
}
\examples{
## create a clusterization
clusters <- paste0("",sample(7, 100, replace = TRUE))
names(clusters) <- paste0("E_",formatC(1:100,  width = 3, flag = "0"))

## create a clusters list from a clusterization
clustersList <- toClustersList(clusters)
head(clustersList, 1)

## recreate the clusterization from the cluster list
clusters2 <- fromClustersList(clustersList, names(clusters))
all.equal(factor(clusters), clusters2)

cl1Size <- length(clustersList[["1"]])

## establish the permutation that groups clusters together
perm <- groupByClusters(clusters)
!is.unsorted(head(names(clusters)[perm],cl1Size))
head(clusters[perm], cl1Size)

## it is possible to have the list of the element names different
## from the names in the clusters list
selectedNames <- paste0("E_",formatC(11:110,  width = 3, flag = "0"))
perm2 <- groupByClustersList(selectedNames, toClustersList(clusters))
all.equal(perm2[91:100], c(91:100))

## is is possible to merge a few clusters together
clustersMerged <- mergeClusters(clusters, names = c("7", "2"),
                                mergedName = "7__2")
sum(table(clusters)[c(2, 7)]) == table(clustersMerged)[["7__2"]]

## it is also possible to do multiple merges at once!
## Note the default new clusters' names
clustersMerged2 <-
  multiMergeClusters(clusters2, namesList = list(c("2", "7"),
                                                 c("1", "3", "5")))
table(clustersMerged2)


}
