## ----setup, include=FALSE-----------------------------------------------------
knitr::opts_chunk$set(echo=TRUE, comment="", dev="png")
library(knitr)

## ----load_library, message=FALSE----------------------------------------------
# We'll also be using some functions from dplyr
# BiocManager::install("pathlinkR", version="devel")
library(dplyr)
library(pathlinkR)

## ----datasets_and_preliminary_volcano_plot, message=FALSE---------------------
## A quick look at the DESeq2 results table
data("exampleDESeqResults")
kable(head(exampleDESeqResults[[1]]))

## Generate a volcano plot from the first data frame, with default thresholds
eruption(
    rnaseqResult=exampleDESeqResults[[1]],
    title=names(exampleDESeqResults[1])
)

## ----plot_fold_changes_I, fig.height=8----------------------------------------
plotFoldChange(
    inputList=exampleDESeqResults,
    pathName="Interferon alpha/beta signaling"
)

## ----ppi_networks, fig.width=10, fig.height=8, warning=FALSE------------------
exNetwork <- ppiBuildNetwork(
    rnaseqResult=exampleDESeqResults[[1]],
    filterInput=TRUE,
    order="zero"
)

ppiPlotNetwork(
    network=exNetwork,
    title=names(exampleDESeqResults)[1],
    fillColumn=LogFoldChange,
    fillType="foldChange",
    label=TRUE,
    labelColumn=hgncSymbol,
    legend=TRUE
)

## ----ppiEnrichNetwork, message=FALSE------------------------------------------
exNetworkPathways <- ppiEnrichNetwork(
    network=exNetwork,
    analysis="hallmark",
    filterResults="default",
    geneUniverse = rownames(exampleDESeqResults[[1]])
)

## ----ppiExtractSubnetwork, warning=FALSE, message=FALSE-----------------------
exSubnetwork <- ppiExtractSubnetwork(
    network=exNetwork,
    pathwayEnrichmentResult=exNetworkPathways,
    pathwayToExtract="INTERFERON GAMMA RESPONSE"
)

ppiPlotNetwork(
    network=exSubnetwork,
    fillType="oneSided",
    fillColumn=degree,
    label=TRUE,
    labelColumn=hgncSymbol,
    legendTitle="Degree"
)

## -----------------------------------------------------------------------------
exampleDESeqResults

## ----sigora_enrichment--------------------------------------------------------
enrichedResultsSigora <- pathwayEnrichment(
    inputList=exampleDESeqResults,
    analysis="sigora",
    filterInput=TRUE,
    gpsRepo="default"
)

## Drop the "genes" column for better visibility
enrichedResultsSigora %>%
    select(!genes) %>%
    head() %>%
    kable()

## ----hallmark_enrichment------------------------------------------------------
enrichedResultsHm <- pathwayEnrichment(
    inputList=exampleDESeqResults,
    analysis="hallmark",
    filterInput=TRUE,
    split=TRUE
)

enrichedResultsHm %>%
    select(!genes) %>%
    head() %>%
    kable()

## ----plot_pathways_I, fig.width=10, fig.height=8------------------------------
pathwayPlots(
    pathwayEnrichmentResults=enrichedResultsSigora,
    columns=2
)

## ----pathway_network_I, fig.width=12, fig.height=10, warning=FALSE, message=FALSE----
data("sigoraDatabase")

pathwayDistancesJaccard <- getPathwayDistances(pathwayData = sigoraDatabase)

startingPathways <- pathnetFoundation(
    mat=pathwayDistancesJaccard,
    maxDistance=0.8
)

# Get the enriched pathways from the "COVID Pos Over Time" comparison
exPathwayNetworkInput <- enrichedResultsSigora %>%
    filter(comparison == "COVID Pos Over Time")

myPathwayNetwork <- pathnetCreate(
    pathwayEnrichmentResult=exPathwayNetworkInput,
    foundation=startingPathways
)

## ----plot_pathnet_I, warning=FALSE, fig.height=7------------------------------
pathnetGGraph(
    myPathwayNetwork,
    labelProp=0.1,
    nodeLabelSize=3,
    nodeLabelOverlaps=8,
    segColour="red",
    themeBaseSize = 12
)

## ----session_information, echo=FALSE------------------------------------------
sessionInfo()

