% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/TEADi_resistance.R
\name{TEADi_resistance}
\alias{TEADi_resistance}
\title{TEAD inhibitor resistance}
\format{
\code{MultiAssayExperiment} obtained from an \code{ArchR} project.
Annotated with the Hg38 genome build.
Contains the following experiments:
\itemize{
\item{\strong{TileMatrix}: SingleCellExperiment with 6068436 rows and 4952 columns}
\item{\strong{GeneScoreMatrix}: SingleCellExperiment with 57765 rows and 4952 columns}
\item{\strong{GeneExpressionMatrix}: SingleCellExperiment with 36451 rows and 4952 columns}
\item{\strong{PeakMatrix}: SingleCellExperiment with 103723 rows and 4952 column}
\item{\strong{MotifMatrix}: SingleCellExperiment with 870 rows and 4952 columns}
\item{\strong{TFPeaksDeviationsMatrix}: SingleCellExperiment with 1269 rows and 4952 columns}
\item{\strong{TF_bindingMatrix}: SingleCellExperiment with 1504 rows and 4952 columns}
}
}
\usage{
TEADi_resistance(
  metadata = FALSE,
  experiments = c("TileMatrix", "GeneScoreMatrix", "GeneExpressionMatrix", "PeakMatrix",
    "MotifMatrix", "TFPeaksDeviationsMatrix", "TF_bindingMatrix")
)
}
\arguments{
\item{metadata}{logical flag specifying whether to return data or metadata only}

\item{experiments}{character vector of matrices to return; see \code{Format}}
}
\value{
\code{MultiAssayExperiment} made up of \code{SingleCellExperiment}s
with assays stored as \code{DelayedMatrix} objects.
If \code{metadata = TRUE}, an \code{ExperimentHub} object listing this data set's metadata.
}
\description{
Single cell multiomics on mesothelioma cell line H226 sensitive and resistant to the TEAD inhibitor GNE-7883
}
\section{Data preparation}{


The following workflow was created based on the original code written by Julien Tremblay (julien.tremblay@contractors.roche.com).
\subsection{1. Initiate \code{ArchR} project}{

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# attach ArchR package
library(ArchR)

# configure ArchR
addArchRGenome("hg38")

# create arrow file from fragment files
## list fragment files
fragments <- <FRAGMENT_FILES>
## assign sample names
names(fragments) <- <SAMPLE_IDs>
## create arrows
createArrowFiles(inputFiles = fragments, sampleNames = names(fragments),
                     minTSS = 4, minFrags = 1000)

# specify output directory
outDir <- <OUTPUT_DIRECTORY>

# locate arrow files
arrows <- <ARROW_FILES>
    
doublet.score <- addDoubletScores(
    input = arrows,
    k = 10, #Refers to how many cells near a 'pseudo-doublet' to count.
    knnMethod = 'UMAP', #Refers to the embedding to use for nearest neighbor search.
    LSIMethod = 1
)

# create ArchR project
project <- ArchRProject(arrows, outDir)
}\if{html}{\out{</div>}}
}

\subsection{2. Add further standard analysis to the ArchR Project}{

This part of code is wrapped by one of the internal libraries.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{
# .h5 files to be downloaded from GEO (GSE247442)
se.rna <- import.archr.10x.se(h5.files = rna.files, sample.names = names(rna.files))

# filter RNA so that only cells which also have ATAC-seq are included
se.rna <- se.rna[,which(colnames(se.rna) \%in\% rownames(ArchR::getCellColData(archr.proj)))]
  
# Filter out scaffolds that don't have at least 3 genes
filtered.chr.names <- names(which(table(seqnames(se.rna)) >= 3));
se.rna <- se.rna[as.character(seqnames(se.rna)) \%in\% filtered.chr.names,];
rowRanges(se.rna) <- keepSeqlevels(rowRanges(se.rna), filtered.chr.names, pruning.mode = 'coarse');

archr.proj <- addGeneExpressionMatrix(
    input = project,
    seRNA = se.rna,
    threads = num.threads
);

# LSI dimensionality reduction
archr.proj <- addIterativeLSI(
    ArchRProj = project,
    useMatrix = 'TileMatrix',
    name = paste0('IterativeLSI_ATAC'),
    seed = 2,
    threads = num.threads
)

# Add clusters to colData
archr.proj <- addClusters(
    input = project,
    reducedDims = paste0('IterativeLSI_ATAC'),
    name = paste0('Clusters',atac.name.suffix),
    seed = 2
)
  
 
}\if{html}{\out{</div>}}
}

\subsection{3. Demultiplex HTO counts}{

\if{html}{\out{<div class="sourceCode r">}}\preformatted{
# HTO counts were generated using internal cumulus cellranger workflow and the results are
# available on Gene Expression Omnibus, GSE247442
# This was equivalent to run in the command line:
#generate_count_matrix_ADTs \\
#    /gstore/apps/CellRanger/7.1.0/lib/python/cellranger/barcodes/737K-arc-v1.txt.gz \\
#    ../data/hashing_index.csv \\
#    ../data/hto/raw_reads_LIB5457060_SAM24417357 \\
#    LIB5457060_SAM24417357 \\
#    -p 4 --max-mismatch-feature 2 --feature antibody --max-mismatch-cell 1 --umi-length 12
#
#generate_count_matrix_ADTs \\
#    /gstore/apps/CellRanger/7.1.0/lib/python/cellranger/barcodes/737K-arc-v1.txt.gz \\
#    ../data/hashing_index.csv \\
#    ../data/hto/raw_reads_LIB5457058_SAM24417355 \\
#    LIB5457058_SAM24417355 \\
#    -p 4 --max-mismatch-feature 2 --feature antibody --max-mismatch-cell 1 --umi-length 12

library(zellkonverter)
library(ArchR)
library(DropletUtils)
library(SingleCellExperiment)
library(Matrix)
library(BiocParallel)
library(data.table)
library(dplyr)

hto_files = list.files(hto_path, pattern="*.hashing.csv", recursive=TRUE, full.names=TRUE)
hto_info2 = data.frame(hto_files)
hto_info2$sample = gsub(".*_(SAM\\\\d+).*", "\\\\1", hto_info2$hto_files)


arcseq_files = list.files(paste0(arcseq_info$uri, "/"), pattern="*raw_feature_bc_matrix", recursive=TRUE, full.names=TRUE)
if(!is.null(skip_sample))\{
    arcseq_files = arcseq_files[!grepl("SAM24417356", arcseq_files)]
\}
arcseq_info2 = data.frame(arcseq_files)
arcseq_info2$sample = gsub(".*(SAM\\\\d+).*", "\\\\1", arcseq_info2$arcseq_files)

samples = gsub(".*_(SAM\\\\d+).*", "\\\\1", hto_files)
summary_df_final = NULL
seRNA_objects = list()
#figures = list()
tables = list()
empty_drops = list()
hashed_drops = list()
for(i in 1:length(samples))\{
        summary_df = NULL
      
        curr_sample = samples[i]
        message("Processing ", curr_sample)
        dir.create(paste0(outdir, "/", curr_sample, "/rds"), recursive=TRUE)
      
        curr_arcseq_file = arcseq_info2[arcseq_info2$sample == curr_sample,]$arcseq_files
        curr_hto_file    = hto_info2[hto_info2$sample == curr_sample,]$hto_files
        ###########################
        # import gex              #
        ###########################
        message("...Importing arcseq matrix")
        seRNA <- ArchR::import10xFeatureMatrix(
            input = file.path(curr_arcseq_file),
            names = curr_sample
        )
        names(assays(seRNA)) = "counts"
        # coerce seRNA obs to a SingleCellExperiment because it is a RangedSingleCellExperiment at this point.
        seRNA = as(seRNA, "SingleCellExperiment")
        # Here if you look at the seRNA obj, there are approx 725k single-cells: at this point, they are NOT single cells, but droplets and the
        # vast majority of them are empty (processed below)
      
        ###########################################
        # import HTO and convert to sce object.   #
        ###########################################
        hto = data.frame(fread(curr_hto_file, header=T), check.names=FALSE)
        rownames_hto = hto$Antibody
        hto = hto[,-1]
        # Here remember that each hto also contains a cell barcode sequence. in the next line,
        # we are formatting each cell barcode the way as it is formatted in the seRNA object
        # will be used later to only keep cell barcodes for which we have an assigned hto.
        colnames(hto) = paste0(curr_sample, "#", colnames(hto), "-1")
        hto = as(as.matrix(hto), "dgCMatrix")
        rownames(hto) = rownames_hto
        hto = SingleCellExperiment(assays = list(counts=hto))
      
        # merge GEX and HTO into a SCE
        # Only keep seRNA cells that are found in hto data.
        common_cells = intersect(colnames(hto), colnames(seRNA))

        # Then, narrow selection to common cells.
        hto = hto[, common_cells] # row=features, col=cells
        # Same selection with seRNA obj (remember both SingleCellExperiment and SummarizedExperiment classes have the same structure)
        seRNA = seRNA[, common_cells]
        # coerce seRNA obs to a SingleCellExperiment because it is a RangedSingleCellExperiment at this point.
        seRNA = as(seRNA, "SingleCellExperiment")
        # Add HTO data as altExp attribute, see doc for more details.
        altExp(seRNA, "HTO") = hto
      
        ###################################################
        # Distinguish single cells from empty droplets.   #
        ###################################################
        # Remove all zeros
        # We can't remove rows at this point, only col (i.e. single cells) at this point. because downstream, we'll have to merge various objects. We'll have to do that later downstream.
        seRNA = seRNA[, Matrix::colSums(assays(seRNA)$counts) > 0]
        # Have a look at barcode ranks:
        empty_thresh = 100 # remove more invalid cells.
        bc_ranks = barcodeRanks(counts(seRNA), lower=empty_thresh)
      
        # Add colData to seRNA obj (i.e. bc_ranks hold barcodes in the same order as it was in the seRNA obj)
        colData(seRNA)$BarcodeRank   = bc_ranks$rank
        colData(seRNA)$BarcodeTotal  = bc_ranks$total
        colData(seRNA)$BarcodeFitted = bc_ranks$fitted
      
      
        # Barcodes that contain more than retain total counts are always retained. This ensures that large cells with profiles that 
        # are very similar to the ambient pool are not inadvertently discarded. If retain is not specified, it is set to the total 
        # count at the knee point detected by barcodeRanks. Manual specification of retain may be useful if the knee point was not correctly 
        # identified in complex log-rank curves. Users can also set retain=Inf to disable automatic retention of barcodes with large totals.
      
        curr_rank = 30000
        emp_drops = emptyDrops(counts(seRNA), lower=NULL, niters=100000,
                             test.ambient = TRUE, BPPARAM=SerialParam(), by.rank=curr_rank)
        emp_fdr = 0.01
        is_cell = emp_drops$FDR <= emp_fdr
        is_cell[is.na(is_cell)] = FALSE # NA means its not a cell, so force NAs to FALSE
        empty_drops[[curr_sample]] = emp_drops
      
        seRNA_filt = runUMAP(seRNA_filt, altexp = "HTO", name="UMAP_HTO", exprs_values = "clr")

        seRNA_filt = seRNA[, which(is_cell)]
        
        # Estimate HTO ambient proportions using empty droplets
        hto_mat = assay(altExp(seRNA),"counts")[, which(is_cell)]
        # confirm that by ambient, we mean non-cells... or just really low abundant stuff.
        # Ok, from the documentation, I believe ambient = non-cells HTOs.
        # Get \% of each HTO
        ambient_hto_mat = assay(altExp(seRNA), "counts")[,!is_cell]
        ambient_hto_prop = proportions(rowSums(assay(altExp(seRNA), "counts")[,!is_cell]))
        hash_stats = hashedDrops(hto_mat, ambient=ambient_hto_prop, 
                               doublet.nmads=3,
                               doublet.min=2,
                               confident.nmads=1,
                               confident.min=1
        )

    
    colData(seRNA_filt) = cbind(colData(seRNA_filt), hash_stats)
    colData(seRNA_filt)$library = sapply(strsplit(colnames(seRNA_filt), split = "#"), "[",1)
  
    assay(altExp(seRNA_filt), "logcounts") = log10(assay(altExp(seRNA_filt), "counts")+1)
    # Here see the doc on sweep. But basically for each row the logcount is subtracted from mean log count, which basically is a clr.
    assay(altExp(seRNA_filt), "clr") = sweep(
                                        assay(altExp(seRNA_filt), "logcounts"), 
                                        2,
                                        colMeans(assay(altExp(seRNA_filt), "logcounts")),
                                        "-"
                                      )
    seRNA_filt = runUMAP(seRNA_filt, altexp = "HTO", name="UMAP_HTO", exprs_values = "clr")
    seRNA_filt$hash_assignment = rownames_hto[seRNA_filt$Best]
    # save output
    saveRDS(seRNA_filt, paste0(outdir, "/", curr_sample, "/rds/", curr_sample, "_demultiplex_hto_and_scRNA.rds"))
\}

}\if{html}{\out{</div>}}
}

\subsection{4. Merge HTO data into the ArchR project}{

\if{html}{\out{<div class="sourceCode r">}}\preformatted{library(ArchR)
library(SingleCellExperiment)
library(data.table)

seRNA_objs = list()
for(sample in samples)\{
    curr_seRNA_obj = readRDS(paste0("./output/", sample, "/rds/", sample, "_demultiplex_hto_and_scRNA.rds"))
    seRNA_objs[[sample]] = curr_seRNA_obj
\}

# Here load the multiple seRNA objects from last step and combine them.
seRNA_final = do.call(cbind, seRNA_objs)

seRNA_final$hash_assignment2 = paste0(seRNA_final$library, "_", seRNA_final$hash_assignment)

common = intersect(project$cellNames, colnames(seRNA_final))
project = project[common,]

# Here at this point, we have the archr object and seRNA obj with the same cells.
# add HTO information
for(curr_cell_attribute in colnames(colData(seRNA_final)))\{
    proj = addCellColData(
        ArchRProj = project,
        data = colData(seRNA_final)[common, curr_cell_attribute],
        cells = common,
        name = curr_cell_attribute,
        force = TRUE
    )
\}

#filter out doublets and non-confident calls
project = project[which(project$Confident == TRUE & proj$Doublet == FALSE), ]

# mapping_file.tsv can be dowloaded from GEO (GSE247442)

mapping = data.frame(fread("mapping_file.tsv", header=T, sep="\\t"), check.names=F)
row.names(mapping) = mapping$SampleID
mapping$hash_assignment2 = mapping$SampleID

tmp_df = data.frame(getCellColData(project, select=c("hash_assignment2")))
tmp_df2 = join(tmp_df, mapping, by="hash_assignment2")
project$CNAME = tmp_df2$CNAME
project$TEST_ARTICLE = tmp_df2$TEST_ARTICLE
project$HTO = tmp_df2$HTO
project$Treatment = paste0(project$CNAME, "_", project$TEST_ARTICLE)
project$Treatment_HTO = paste0(project$CNAME, "_", project$TEST_ARTICLE, " (", project$HTO, ")")
project$Treatment_library = paste0(project$Treatment, "_", project$library)

}\if{html}{\out{</div>}}
}

\subsection{5. Cluster annotation}{

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# mapping_file.tsv can be dowloaded from GEO (GSE247442)
mapping = data.frame(fread("mapping_file.tsv", header=T, sep="\\t"), check.names=F)
samples_vColors = vColors[1:length(unique(mapping[["Treatment_HTO"]]))]
names(samples_vColors) = unique(mapping[["Treatment_HTO"]])

embed = getEmbedding(project, embedding=paste0("UMAP_TileMatrix"))
embed = cbind(embed, getCellColData(project, select=c("Treatment", "HTO", "Treatment_HTO", "Treatment_library", "library", "Clusters_TileMatrix")))
colnames(embed)[1] = "UMAP_dim_1"
colnames(embed)[2] = "UMAP_dim_2"

my_labels_df = data.frame(Treatment_HTO=unique(embed$Treatment_HTO),label=paste0("", unique(embed$Treatment_HTO)))
my_labels = embed \%>\%
    dplyr::group_by(Treatment_HTO) \%>\%
    dplyr::summarize(UMAP_dim_1 = median(UMAP_dim_1), UMAP_dim_2 = median(UMAP_dim_2)) \%>\%
    dplyr::left_join(my_labels_df) \%>\% as.data.frame()

p_init_samples = NULL; p_init_samples = ggplot(embed, aes(x=UMAP_dim_1, y=UMAP_dim_2, color=Treatment_HTO)) +
    geom_point(size=0.3) +
    scale_color_manual(values=samples_vColors) +
    xlab(paste0("UMAP 1")) +
    ylab(paste0("UMAP 2")) +
    #ggtitle(paste0("UMAP; ", curr_signature)) +
    guides(color=guide_legend(ncol=2, override.aes=list(size=1.5))) +
    ggrepel::geom_label_repel(data=my_labels, box.padding=0.0, alpha=0.8, label.size=0.05, fill="gray70", color="black", size=3, aes(label=my_labels[["label"]])) +
    theme_minimal() +
    theme(panel.border=element_rect(fill=NA, linetype="solid", colour = "black", linewidth=0.5), axis.title=element_text(size=8), 
          axis.text=element_text(size=8, angle=0), strip.text.y=element_text(size=8, angle=0), legend.position="bottom",
          legend.text=element_text(size=8, angle=0), plot.title=element_text(size=10), legend.key.size=unit(0.35, 'cm'), legend.title=element_blank())

my_labels_df2 = data.frame(Clusters_TileMatrix=unique(embed$Clusters_TileMatrix),label=paste0("", unique(embed$Clusters_TileMatrix)))
my_labels2 = embed \%>\%
    dplyr::group_by(Clusters_TileMatrix) \%>\%
    dplyr::summarize(UMAP_dim_1 = median(UMAP_dim_1), UMAP_dim_2 = median(UMAP_dim_2)) \%>\%
    dplyr::left_join(my_labels_df2) \%>\% as.data.frame()

clusters_vColors = vColors20[1:length(unique(mapping[["Clusters_TileMatrix"]]))]
names(clusters_vColors) = unique(mapping[["Clusters_TileMatrix"]])

p_init_clusters = NULL; p_init_clusters = ggplot(embed, aes(x=UMAP_dim_1, y=UMAP_dim_2, color=.data[["Clusters_TileMatrix"]])) +
    geom_point(size=0.3) +
    scale_color_manual(values=clusters_vColors) +
    xlab(paste0("UMAP 1")) +
    ylab(paste0("UMAP 2")) +
    #ggtitle(paste0("UMAP; ", curr_signature)) +
    guides(color = guide_legend(override.aes=list(size=1.5))) + guides(fill=guide_legend(ncol=3)) + #guides(fill = guide_legend(override.aes = list(size=4))) +
    ggrepel::geom_label_repel(data=my_labels2, fill="gray70", box.padding=0.0, alpha=0.8, label.size=0.05, color="black", size=3, aes(label=my_labels2[["label"]])) +
    theme_minimal() +
    theme(panel.border=element_rect(fill=NA, linetype="solid", colour = "black", linewidth=0.5), axis.title=element_text(size=8), 
          axis.text=element_text(size=8, angle=0), strip.text.y=element_text(size=8, angle=0), legend.position="bottom",
          legend.text=element_text(size=8, angle=0), plot.title=element_text(size=10), legend.key.size=unit(0.35, 'cm'), legend.title=element_blank())
#p_init_clusters
p1 = ggarrange(p_init_samples, p_init_clusters)
#ggsave(paste0(curr_outdir, "/marker_UMAP_TilleMatrix_sample_vs_cluster_ATAC", ".pdf"), plot=p1, device="pdf", height=5.11, width=9.5, units="in", limitsize=FALSE)
#ggsave(paste0(curr_outdir, "/marker_UMAP_TileMatrix_sample_vs_cluster_ATAC", ".png"), plot=p1, device="png", height=5.11, width=9.5, units="in", limitsize=FALSE)

# C1 = Sensitive_common_small;   Sensitive_side
# C2 = Sensitive_no_agent_big    Sensitive_DMSO
# C3 = Sensitive_agent;          Sensitive_GNE7883
# C4 = Sensitive_agent;          Sensitive_GNE7883
# C5 = Resistant_small;          Resistant_side
# C6 = Resistant_big;            Resistant_main
# C7 = Resistant_big;            Resistant_main 
#From XY: resistant_main (big), resistant_side (small), sensitive_DMSO, sensitive_GNE7883, sensitive_side (common)
project$Clusters_TileMatrix_named = plyr::mapvalues(project$Clusters_TileMatrix,
                                from = paste0("C_TileMatrix",1:7),
                                to = c("Sensitive_side", "Sensitive_DMSO", "Sensitive_GNE7883", 
                                       "Sensitive_GNE7883",  "Resistant_side", "Resistant_main", 
                                       "Resistant_main")
)
}\if{html}{\out{</div>}}
}

\subsection{6. Generate ordinations}{

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# add reduced dims TileMatrix
project = addIterativeLSI(project, useMatrix='TileMatrix', name='IterativeLSI_TileMatrix', seed=2, force=TRUE, threads=num_threads)
project = addClusters(project, reducedDims='IterativeLSI_TileMatrix', name='Clusters_TileMatrix', seed=2, prefix='C_TileMatrix', force=TRUE, threads=num_threads)
project = addUMAP(project, reducedDims='IterativeLSI_TileMatrix', name='UMAP_TileMatrix', seed=2, force=TRUE, threads=num_threads)

# add reduced dims GeneExpressionMatrix
project = addIterativeLSI(project, useMatrix='GeneExpressionMatrix', name='IterativeLSI_GeneExpressionMatrix', seed=2, force=TRUE, threads=num_threads)
project = addClusters(project, reducedDims='IterativeLSI_GeneExpressionMatrix', name='Clusters_GeneExpressionMatrix', seed=2, force=TRUE, prefix='C_GeneExpressionMatrix', threads=num_threads)
project = addUMAP(project, reducedDims='IterativeLSI_GeneExpressionMatrix', name='UMAP_GeneExpressionMatrix', seed=2, force=TRUE, threads=num_threads)

# Combine both dimensionalities, add cluster to resulting combined dim and add umap
project = addCombinedDims(project, reducedDims=c('IterativeLSI_TileMatrix', 'IterativeLSI_GeneExpressionMatrix'), name='IterativeLSI_Combined')
project = addClusters(project, reducedDims='IterativeLSI_Combined', name='Clusters_Combined', seed=2, force=TRUE, prefix='C_Combined', threads=num_threads)
project = addUMAP(project, reducedDims='IterativeLSI_Combined', name='UMAP_Combined', seed=2, force=TRUE, threads=num_threads)

# Add tSNE embeddings
project = addTSNE(project, reducedDims='IterativeLSI_TileMatrix', name='TSNE_TileMatrix', perplexity=30, seed=2, force=TRUE, threads=num_threads)
project = addTSNE(project, reducedDims='IterativeLSI_GeneExpressionMatrix', name='TSNE_GeneExpressionMatrix', perplexity=30, seed=2, force=TRUE, threads=num_threads)
project = addTSNE(project, reducedDims='IterativeLSI_Combined', name='TSNE_Combined', perplexity=30, seed=2, force=TRUE, threads=num_threads)
project = addTSNE(project, reducedDims='IterativeLSI_ATAC', name='TSNE_ATAC', perplexity=30, seed=2, force=TRUE, threads=num_threads)
}\if{html}{\out{</div>}}
}

\subsection{7. Peak calling}{

\if{html}{\out{<div class="sourceCode r">}}\preformatted{library(MASS)
library(chromVARmotifs)

project = addGroupCoverages(project, groupBy='Clusters_TileMatrix', threads=num_threads)

project = addReproduciblePeakSet(project, groupBy="Clusters_TileMatrix", method="q", cutOff=0.05,
                                    pathToMacs2="/gstore/home/tremblj2/software/macs2/macs2_venv/bin/macs2", 
                                    excludeChr=c('chrMT','chrY'), genomeSize=2.7e9, threads=num_threads)

project = addPeakMatrix(project, binarize=FALSE, threads=num_threads, force=TRUE)

# TF annotation
peaks_anno = genomitory::getFeatures('GMTY162:hg38_motif_bed_granges@REVISION-3')
project = addPeakAnnotations(project, regions=peaks_anno, name='ENCODE_and_cistromeDB_TF_peaks', force=TRUE)
project = addDeviationsMatrix(project, peakAnnotation='ENCODE_and_cistromeDB_TF_peaks', 
                                 matrixName='TFPeaksDeviationsMatrix', threads=num_threads, force=TRUE)

# motif annotation
project = addMotifAnnotations(project, motifSet='cisbp', annoName='Motif', species='Homo sapiens', force=TRUE)
project = addDeviationsMatrix(project, peakAnnotation='Motif', threads=num_threads, force=TRUE)

# TF binding matrix 
project = addPeakAnnotations(project, regions=epiregulon::getTFMotifInfo(genome="hg38"), name="TF_binding", force=T, logFile="addPeakAnnotations")
project = addDeviationsMatrix(project, peakAnnotation="TF_binding", threads=num_threads)
}\if{html}{\out{</div>}}
}
}

\section{Data storage and access}{

The \code{MultiAssayExperiments} is split into separate \code{SingleCellExperiment}
objects and they in turn are split into components, all of which are stored in a
single hdf5 file. Data and can be accessed with a special function that extracts
elements of the requested experiment(s), reassembles them, and builds an MAE.
}

\examples{
# check metada of dataset
TEADi_resistance(metadata = TRUE)

# download data
 \dontrun{
TEADi_resistance()
}

}
\references{
Manuscript under review...
}
