#' @title Example TENET MultiAssayExperiment object
#'
#' @description A MultiAssayExperiment dataset created using a modified version
#' of the `TCGADownloader` function from the TENET package utilizing
#' TCGAbiolinks package functionality. This object contains two
#' SummarizedExperiment objects, `expression` and `methylation`, with
#' expression data for 11,637 genes annotated to the GENCODE v36 dataset,
#' including all 1,637 identified human TF genes, and DNA methylation data for
#' 20,000 probes from the Illumina HM450 methylation array. The data are
#' aligned to the human hg38 genome. Expression and methylation values were
#' matched from 200 tumor and 42 adjacent normal tissue samples subset from the
#' TCGA BRCA dataset. Additionally, results from running the TENET step 1-6
#' functions on these samples are included in the metadata of this
#' MultiAssayExperiment object. Clinical data for these samples are included in
#' the colData of the MultiAssayExperiment object. (A separate data frame
#' object containing a subset of the clinical data for these samples is
#' available as `exampleTENETClinicalDataFrame`.) This dataset is included to
#' demonstrate TENET functions. Note: Because this dataset is a small subset of
#' the overall BRCA dataset, results generated by TENET from this dataset
#' differ from those presented for the BRCA dataset at large in TENET
#' publications.
#'
#' @param metadata If TRUE, retrieve the ExperimentHub metadata instead of the
#' object itself. Defaults to FALSE.
#'
#' @examplesIf interactive()
#' exampleTENETMultiAssayExperiment()
#'
#' @export
#'
#' @return A MultiAssayExperiment object composed of "expression" and
#' "methylation" SummarizedExperiment objects, clinical information about
#' the patients in the colData, and information about sample pairing and
#' type ('Case' vs. 'Control') in the sampleMap.
exampleTENETMultiAssayExperiment <- function(metadata = FALSE) {
    .getExperimentHubDataset(
        "exampleTENETMultiAssayExperiment",
        metadata = metadata
    )
}

#' @title Example TENET clinical data frame
#'
#' @description A data frame containing example and simulated clinical
#' information corresponding to the samples in the
#' `exampleTENETMultiAssayExperiment` object, used to demonstrate how TENET
#' functions can import clinical data from a specified data frame. Clinical
#' data are utilized by the `step2GetDifferentiallyMethylatedSites`,
#' `step7TopGenesSurvival`, and `step7ExpressionVsDNAMethylationScatterplots`
#' functions. The data frame consists of vital status and time variables for
#' use by the `step7TopGenesSurvival` function, simulated purity data for each
#' sample, and simulated copy number variation (CNV) and somatic mutation (SM)
#' data for the top 10 genes by number of linked hypermethylated and
#' hypomethylated probes derived from analyses done using the
#' `exampleTENETMultiAssayExperiment` object. These data are a subset of the
#' clinical data contained in the colData of the
#' `exampleTENETMultiAssayExperiment` object.
#'
#' @param metadata If TRUE, retrieve the ExperimentHub metadata instead of the
#' object itself. Defaults to FALSE.
#'
#' @examplesIf interactive()
#' exampleTENETClinicalDataFrame()
#'
#' @export
#'
#' @return A data frame with 231 rows and 43 variables.
#' \describe{
#'   \item{\code{vital_status}}{(character) Contains vital status data for each
#'     sample, listed as either "Alive", "Dead", or NA. Values are derived
#'     directly from the TCGA patient data.
#'   }
#'   \item{\code{time}}{(double) Contains survival time values, in days, for
#'     each sample. Values are derived directly from the TCGA patient data.
#'   }
#'   \item{\code{purity}}{(double) Contains simulated purity data for each
#'     sample, ranging from 0 to 1.
#'   }
#'   \item{\code{ENSG00000165821_CNV}}{
#'     (integer) Contains simulated CNV data for each sample for the
#'     ENSG00000165821 (SALL2) gene, ranging from -2 to 2.
#'   }
#'   \item{\code{ENSG00000165821_SM}}{
#'     (integer) Contains simulated SM data for each sample for the
#'     ENSG00000165821 (SALL2) gene, either 0 or 1.
#'   }
#'   \item{\code{ENSG00000169989_CNV}}{
#'     (integer) Contains simulated CNV data for each sample for the
#'     ENSG00000169989 (TIGD4) gene, ranging from -2 to 2.
#'   }
#'   \item{\code{ENSG00000169989_SM}}{
#'     (integer) Contains simulated SM data for each sample for the
#'     ENSG00000169989 (TIGD4) gene, either 0 or 1.
#'   }
#'   \item{\code{ENSG00000197343_CNV}}{
#'     (integer) Contains simulated CNV data for each sample for the
#'     ENSG00000197343 (ZNF655) gene, ranging from -2 to 2.
#'   }
#'   \item{\code{ENSG00000197343_SM}}{
#'     (integer) Contains simulated SM data for each sample for the
#'     ENSG00000197343 (ZNF655) gene, either 0 or 1.
#'   }
#'   \item{\code{ENSG00000169083_CNV}}{
#'     (integer) Contains simulated CNV data for each sample for the
#'     ENSG00000169083 (AR) gene, ranging from -2 to 2.
#'   }
#'   \item{\code{ENSG00000169083_SM}}{
#'     (integer) Contains simulated SM data for each sample for the
#'     ENSG00000169083 (AR) gene, either 0 or 1.
#'   }
#'   \item{\code{ENSG00000177842_CNV}}{
#'     (integer) Contains simulated CNV data for each sample for the
#'     ENSG00000177842 (ZNF620) gene, ranging from -2 to 2.
#'   }
#'   \item{\code{ENSG00000177842_SM}}{
#'     (integer) Contains simulated SM data for each sample for the
#'     ENSG00000177842 (ZNF620) gene, either 0 or 1.
#'   }
#'   \item{\code{ENSG00000234284_CNV}}{
#'     (integer) Contains simulated CNV data for each sample for the
#'     ENSG00000234284 (ZNF879) gene, ranging from -2 to 2.
#'   }
#'   \item{\code{ENSG00000234284_SM}}{
#'     (integer) Contains simulated SM data for each sample for the
#'     ENSG00000234284 (ZNF879) gene, either 0 or 1.
#'   }
#'   \item{\code{ENSG00000177853_CNV}}{
#'     (integer) Contains simulated CNV data for each sample for the
#'     ENSG00000177853 (ZNF518A) gene, ranging from -2 to 2.
#'   }
#'   \item{\code{ENSG00000177853_SM}}{
#'     (integer) Contains simulated SM data for each sample for the
#'     ENSG00000177853 (ZNF518A) gene, either 0 or 1.
#'   }
#'   \item{\code{ENSG00000196345_CNV}}{
#'     (integer) Contains simulated CNV data for each sample for the
#'     ENSG00000196345 (ZKSCAN7) gene, ranging from -2 to 2.
#'   }
#'   \item{\code{ENSG00000196345_SM}}{
#'     (integer) Contains simulated SM data for each sample for the
#'     ENSG00000196345 (ZKSCAN7) gene, either 0 or 1.
#'   }
#'   \item{\code{ENSG00000196653_CNV}}{
#'     (integer) Contains simulated CNV data for each sample for the
#'     ENSG00000196653 (ZNF502) gene, ranging from -2 to 2.
#'   }
#'   \item{\code{ENSG00000196653_SM}}{
#'     (integer) Contains simulated SM data for each sample for the
#'     ENSG00000196653 (ZNF502) gene, either 0 or 1.
#'   }
#'   \item{\code{ENSG00000162599_CNV}}{
#'     (integer) Contains simulated CNV data for each sample for the
#'     ENSG00000162599 (NFIA) gene, ranging from -2 to 2.
#'   }
#'   \item{\code{ENSG00000162599_SM}}{
#'     (integer) Contains simulated SM data for each sample for the
#'     ENSG00000162599 (NFIA) gene, either 0 or 1.
#'   }
#'   \item{\code{ENSG00000129514_CNV}}{
#'     (integer) Contains simulated CNV data for each sample for the
#'     ENSG00000129514 (FOXA1) gene, ranging from -2 to 2.
#'   }
#'   \item{\code{ENSG00000129514_SM}}{
#'     (integer) Contains simulated SM data for each sample for the
#'     ENSG00000129514 (FOXA1) gene, either 0 or 1.
#'   }
#'   \item{\code{ENSG00000124664_CNV}}{
#'     (integer) Contains simulated CNV data for each sample for the
#'     ENSG00000124664 (SPDEF) gene, ranging from -2 to 2.
#'   }
#'   \item{\code{ENSG00000124664_SM}}{
#'     (integer) Contains simulated SM data for each sample for the
#'     ENSG00000124664 (SPDEF) gene, either 0 or 1.
#'   }
#'   \item{\code{ENSG00000107485_CNV}}{
#'     (integer) Contains simulated CNV data for each sample for the
#'     ENSG00000107485 (GATA3) gene, ranging from -2 to 2.
#'   }
#'   \item{\code{ENSG00000107485_SM}}{
#'     (integer) Contains simulated SM data for each sample for the
#'     ENSG00000107485 (GATA3) gene, either 0 or 1.
#'   }
#'   \item{\code{ENSG00000091831_CNV}}{
#'     (integer) Contains simulated CNV data for each sample for the
#'     ENSG00000091831 (ESR1) gene, ranging from -2 to 2.
#'   }
#'   \item{\code{ENSG00000091831_SM}}{
#'     (integer) Contains simulated SM data for each sample for the
#'     ENSG00000091831 (ESR1) gene, either 0 or 1.
#'   }
#'   \item{\code{ENSG00000118513_CNV}}{
#'     (integer) Contains simulated CNV data for each sample for the
#'     ENSG00000118513 (MYB) gene, ranging from -2 to 2.
#'   }
#'   \item{\code{ENSG00000118513_SM}}{
#'     (integer) Contains simulated SM data for each sample for the
#'     ENSG00000118513 (MYB) gene, either 0 or 1.
#'   }
#'   \item{\code{ENSG00000100219_CNV}}{
#'     (integer) Contains simulated CNV data for each sample for the
#'     ENSG00000100219 (XBP1) gene, ranging from -2 to 2.
#'   }
#'   \item{\code{ENSG00000100219_SM}}{
#'     (integer) Contains simulated SM data for each sample for the
#'     ENSG00000100219 (XBP1) gene, either 0 or 1.
#'   }
#'   \item{\code{ENSG00000152192_CNV}}{
#'     (integer) Contains simulated CNV data for each sample for the
#'     ENSG00000152192 (POU4F1) gene, ranging from -2 to 2.
#'   }
#'   \item{\code{ENSG00000152192_SM}}{
#'     (integer) Contains simulated SM data for each sample for the
#'     ENSG00000152192 (POU4F1) gene, either 0 or 1.
#'   }
#'   \item{\code{ENSG00000105261_CNV}}{
#'     (integer) Contains simulated CNV data for each sample for the
#'     ENSG00000105261 (OVOL3) gene, ranging from -2 to 2.
#'   }
#'   \item{\code{ENSG00000105261_SM}}{
#'     (integer) Contains simulated SM data for each sample for the
#'     ENSG00000105261 (OVOL3) gene, either 0 or 1.
#'   }
#'   \item{\code{ENSG00000178935_CNV}}{
#'     (integer) Contains simulated CNV data for each sample for the
#'     ENSG00000178935 (ZNF552) gene, ranging from -2 to 2.
#'   }
#'   \item{\code{ENSG00000178935_SM}}{
#'     (integer) Contains simulated SM data for each sample for the
#'     ENSG00000178935 (ZNF552) gene, either 0 or 1.
#'   }
#'   \item{\code{ENSG00000115163_CNV}}{
#'     (integer) Contains simulated CNV data for each sample for the
#'     ENSG00000115163 (CENPA) gene, ranging from -2 to 2.
#'   }
#'   \item{\code{ENSG00000115163_SM}}{
#'     (integer) Contains simulated SM data for each sample for the
#'     ENSG00000115163 (CENPA) gene, either 0 or 1.
#'   }
#' }
exampleTENETClinicalDataFrame <- function(metadata = FALSE) {
    .getExperimentHubDataset(
        "exampleTENETClinicalDataFrame",
        metadata = metadata
    )
}

#' @title Example TENET `step1MakeExternalDatasets` GRanges object
#'
#' @description A GenomicRanges dataset representing putative enhancer regions
#' relevant to BRCA, created using the `step1MakeExternalDatasets` function in
#' the TENET package with the `consensusEnhancer`, `consensusNDR`,
#' `publicEnhancer`, `publicNDR`, and `ENCODEdELS` arguments all set to TRUE,
#' and the `cancerType` argument set to "BRCA". The data are aligned to the
#' human hg38 genome. This dataset is included to demonstrate TENET's
#' `step2GetDifferentiallyMethylatedSites` function.
#'
#' @param metadata If TRUE, retrieve the ExperimentHub metadata instead of the
#' object itself. Defaults to FALSE.
#'
#' @examplesIf interactive()
#' exampleTENETStep1MakeExternalDatasetsGRanges()
#'
#' @export
#'
#' @return A GRanges object with 1,971,031 ranges and no metadata.
exampleTENETStep1MakeExternalDatasetsGRanges <- function(metadata = FALSE) {
    .getExperimentHubDataset(
        "exampleTENETStep1MakeExternalDatasetsGRanges",
        metadata = metadata
    )
}

#' @title Example TENET `step2GetDifferentiallyMethylatedSites` purity
#' SummarizedExperiment object
#'
#' @description A SummarizedExperiment object with three DNA methylation
#' datasets each composed of 10 adjacent normal colorectal adenocarcinoma
#' (COAD) samples from The Cancer Genome Atlas (TCGA), retrieved using the
#' TCGAbiolinks package. Each dataset has data for 20,000 probes from the
#' Illumina HM450 methylation array, to match the number of probes in the
#' `exampleTENETMultiAssayExperiment` object. The data are aligned to the human
#' hg38 genome. This object is representative of a `purity` dataset, which
#' would contain DNA methylation data from potentially confounding sources,
#' used with TENET's `step2GetDifferentiallyMethylatedSites` function.
#'
#' @param metadata If TRUE, retrieve the ExperimentHub metadata instead of the
#' object itself. Defaults to FALSE.
#'
#' @examplesIf interactive()
#' exampleTENETStep2GetDifferentiallyMethylatedSitesPuritySummarizedExperiment()
#'
#' @export
#'
#' @return A SummarizedExperiment object with two methylation datasets,
#' "purityMethylationExampleA" and "purityMethylationExampleB", each containing
#' data for 20,000 HM450 probes in 10 samples.
exampleTENETStep2GetDifferentiallyMethylatedSitesPuritySummarizedExperiment <-
    function(metadata = FALSE) {
        .getExperimentHubDataset(
            paste0(
                "exampleTENETStep2GetDifferentiallyMethylatedSites",
                "PuritySummarizedExperiment"
            ),
            metadata = metadata
        )
    }

#' @title Example TENET peak regions
#'
#' @description A GenomicRanges dataset with example genomic regions (peaks) of
#' interest, used to demonstrate TENET's `step7TopGenesUserPeakOverlap`
#' function. The peaks are derived from a ChIP-seq experiment on FOXA1 in MCF-7
#' cells and aligned to the human hg38 genome. They were downloaded from the
#' ENCODE portal (file ENCFF112JVK in experiment ENCSR126YEB). **Citation:**
#' ENCODE Project Consortium; Moore JE, Purcaro MJ, Pratt HE, et al. Expanded
#' encyclopaedias of DNA elements in the human and mouse genomes. Nature. 2020
#' Jul;583(7818):699-710. doi: 10.1038/s41586-020-2493-4. Epub 2020 Jul 29.
#' Erratum in: Nature. 2022 May;605(7909):E3. PMID: 32728249; PMCID: PMC7410828.
#'
#' @param metadata If TRUE, retrieve the ExperimentHub metadata instead of the
#' object itself. Defaults to FALSE.
#'
#' @examplesIf interactive()
#' exampleTENETPeakRegions()
#'
#' @export
#'
#' @return A GRanges object with 37,386 ranges and no metadata.
#'
#' @source{
#'     \url{https://www.encodeproject.org/files/ENCFF112JVK/}
#' }
exampleTENETPeakRegions <- function(metadata = FALSE) {
    .getExperimentHubDataset("exampleTENETPeakRegions", metadata = metadata)
}

#' @title Example TENET topologically associating domain (TAD) regions
#'
#' @description A GenomicRanges dataset with example topologically associating
#' domains (TADs), used to demonstrate TENET's `step7TopGenesTADTables`
#' function. The TADs are derived from T47D cells (mistakenly labeled as
#' 'T470'), and aligned to the human hg38 genome. They were downloaded from the
#' 3D Genome Browser at <http://3dgenome.fsm.northwestern.edu>. **Citation:**
#' Wang Y, Song F, Zhang B, et al. The 3D Genome Browser: a web-based browser
#' for visualizing 3D genome organization and long-range chromatin interactions.
#' Genome Biol. 2018 Oct 4;19(1):151. doi: 10.1186/s13059-018-1519-9. PMID:
#' 30286773; PMCID: PMC6172833.
#'
#' @param metadata If TRUE, retrieve the ExperimentHub metadata instead of the
#' object itself. Defaults to FALSE.
#'
#' @examplesIf interactive()
#' exampleTENETTADRegions()
#'
#' @export
#'
#' @return A GRanges object with 1,889 ranges and no metadata.
#'
#' @source{
#'     \url{http://3dgenome.fsm.northwestern.edu/publications.html}
#' }
exampleTENETTADRegions <- function(metadata = FALSE) {
    .getExperimentHubDataset("exampleTENETTADRegions", metadata = metadata)
}
