% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/AllGenerics.R, R/XcmsExperiment.R,
%   R/methods-XCMSnExp.R
\name{featureChromatograms}
\alias{featureChromatograms}
\alias{featureChromatograms,XcmsExperiment-method}
\alias{featureChromatograms,XCMSnExp-method}
\title{Extract ion chromatograms for each feature}
\usage{
featureChromatograms(object, ...)

\S4method{featureChromatograms}{XcmsExperiment}(
  object,
  expandRt = 0,
  expandMz = 0,
  aggregationFun = "max",
  features = character(),
  return.type = "XChromatograms",
  chunkSize = 2L,
  mzmin = min,
  mzmax = max,
  rtmin = min,
  rtmax = max,
  ...,
  progressbar = TRUE,
  BPPARAM = bpparam()
)

\S4method{featureChromatograms}{XCMSnExp}(
  object,
  expandRt = 0,
  aggregationFun = "max",
  features,
  include = c("feature_only", "apex_within", "any", "all"),
  filled = FALSE,
  n = length(fileNames(object)),
  value = c("maxo", "into"),
  expandMz = 0,
  ...
)
}
\arguments{
\item{object}{\code{XcmsExperiment} or \code{XCMSnExp} object with grouped
chromatographic peaks.}

\item{...}{optional arguments to be passed along to the \code{\link[=chromatogram]{chromatogram()}}
function.}

\item{expandRt}{\code{numeric(1)} to expand the retention time range for each
chromatographic peak by a constant value on each side.}

\item{expandMz}{\code{numeric(1)} to expand the m/z range for each chromatographic
peak by a constant value on each side. Be aware that by extending the
m/z range the extracted EIC might \strong{no longer} represent the actual
identified chromatographic peak because intensities of potential
additional mass peaks within each spectra would be aggregated into the
final reported intensity value per spectrum (retention time).}

\item{aggregationFun}{\code{character(1)} specifying the name that should be
used to aggregate intensity values across the m/z value range for
the same retention time. The default \code{"max"} returns a base peak
chromatogram.}

\item{features}{\code{integer}, \code{character} or \code{logical} defining a subset of
features for which chromatograms should be returned. Can be the index
of the features in \code{featureDefinitions}, feature IDs (row names of
\code{featureDefinitions}) or a logical vector.}

\item{return.type}{\code{character(1)} defining how the result should be
returned. At present only \code{return.type = "XChromatograms"} is
supported and the results are thus returned as an \code{\link[=XChromatograms]{XChromatograms()}}
object.}

\item{chunkSize}{For \code{object} being an \code{XcmsExperiment}: \code{integer(1)}
defining the number of files from which the data should be loaded at
a time into memory. Defaults to \code{chunkSize = 2L}.}

\item{mzmin}{\code{function} defining how the lower boundary of the m/z region
from which the EIC is integrated should be defined. Defaults to
\code{mzmin = min} thus the smallest \code{"mzmin"} value for all chromatographic
peaks of a feature will be used.}

\item{mzmax}{\code{function} defining how the upper boundary of the m/z region
from which the EIC is integrated should be defined. Defaults to
\code{mzmax = max} thus the largest \code{"mzmax"} value for all chromatographic
peaks of a feature will be used.}

\item{rtmin}{\code{function} defining how the lower boundary of the rt region
from which the EIC is integrated should be defined. Defaults to
\code{rtmin = min} thus the smallest \code{"rtmin"} value for all chromatographic
peaks of a feature will be used.}

\item{rtmax}{\code{function} defining how the upper boundary of the rt region
from which the EIC is integrated should be defined. Defaults to
\code{rtmax = max} thus the largest \code{"rtmax"} value for all chromatographic
peaks of a feature will be used.}

\item{progressbar}{\code{logical(1)} defining whether a progress bar is shown.}

\item{BPPARAM}{For \code{object} being an \code{XcmsExperiment}: parallel processing
setup. Defaults to \code{BPPARAM = bpparam()}. See \code{\link[BiocParallel:register]{BiocParallel::bpparam()}}
for more information.}

\item{include}{Only for \code{object} being an \code{XCMSnExp}: \code{character(1)}
defining which chromatographic peaks (and related feature definitions)
should be included in the returned \code{\link[=XChromatograms]{XChromatograms()}}.
Defaults to \code{"feature_only"}; See description above for options and
details.}

\item{filled}{Only for \code{object} being an \code{XCMSnExp}: \code{logical(1)} whether
filled-in peaks should be included in the result object. The default
is \code{filled = FALSE}, i.e. only detected peaks are reported.}

\item{n}{Only for \code{object} being an \code{XCMSnExp}: \code{integer(1)} to optionally
specify the number of \emph{top n} samples from which the EIC should be
extracted.}

\item{value}{Only for \code{object} being an \code{XCMSnExp}: \code{character(1)}
specifying the column to be used to sort the samples. Can be either
\code{"maxo"} (the default) or \code{"into"} to use the maximal peak intensity
or the integrated peak area, respectively.}
}
\value{
\code{\link[=XChromatograms]{XChromatograms()}} object. In future, depending on parameter
\code{return.type}, the data might be returned as a different object.
}
\description{
Extract ion chromatograms for features in an \link{XcmsExperiment} or
\linkS4class{XCMSnExp} object. The function returns for each feature the
extracted ion chromatograms (along with all associated chromatographic
peaks) in each sample. The chromatogram is extracted from the m/z - rt
region that includes \strong{all} chromatographic peaks of a feature. By default,
this region is defined using the range of the chromatographic peaks' m/z
and retention times (with \code{mzmin = min}, \code{mzmax = max}, \code{rtmin = min} and
\code{rtmax = max}). For some features, and depending on the data, the m/z and
rt range can thus be relatively large. The boundaries of the m/z - rt
region can also be restricted by changing parameters \code{mzmin}, \code{mzmax},
\code{rtmin} and \code{rtmax} to a different functions, such as \code{median}.

By default only chromatographic peaks associated with a feature are
included in the returned \link{XChromatograms} object. For \code{object} being an
\code{XCMSnExp} object parameter \code{include} allows also to return all
chromatographic peaks with their apex position within the selected
region (\code{include = "apex_within"}) or any chromatographic peak overlapping
the m/z and retention time range (\code{include = "any"}).
}
\note{
The EIC data of a feature is extracted from every sample using the same
m/z - rt area. The EIC in a sample does thus not exactly represent the
signal of the actually identified chromatographic peak in that sample.
The \code{\link[=chromPeakChromatograms]{chromPeakChromatograms()}} function would allow to extract the actual
EIC of the chromatographic peak in a specific sample. See also examples
below.

Parameters \code{include}, \code{filled}, \code{n} and \code{value} are only supported
for \code{object} being an \code{XCMSnExp}.

When extracting EICs from only the top \code{n} samples it can happen that one
or more of the features specified with \code{features} are dropped because they
have no detected peak in the \emph{top n} samples. The chance for this to happen
is smaller if \code{x} contains also filled-in peaks (with \code{fillChromPeaks}).
}
\examples{

## Load a test data set with detected peaks
library(xcms)
library(MsExperiment)
faahko_sub <- loadXcmsData("faahko_sub2")

## Disable parallel processing for this example
register(SerialParam())

## Perform correspondence analysis
xdata <- groupChromPeaks(faahko_sub,
    param = PeakDensityParam(minFraction = 0.8, sampleGroups = rep(1, 3)))

## Get the feature definitions
featureDefinitions(xdata)

## Extract ion chromatograms for the first 3 features. Parameter
## `features` can be either the feature IDs or feature indices.
chrs <- featureChromatograms(xdata,
    features = rownames(featureDefinitions)[1:3])

## Plot the EIC for the first feature using different colors for each file.
plot(chrs[1, ], col = c("red", "green", "blue"))

## The EICs for all 3 samples use the same m/z and retention time range,
## which was defined using the `featureArea` function:
featureArea(xdata, features = rownames(featureDefinitions(xdata))[1:3],
    mzmin = min, mzmax = max, rtmin = min, rtmax = max)

## To extract the actual (exact) EICs for each chromatographic peak of
## a feature in each sample, the `chromPeakChromatograms` function would
## need to be used instead. Below we extract the EICs for all
## chromatographic peaks of the first feature. We need to first get the
## IDs of all chromatographic peaks assigned to the first feature:
peak_ids <- rownames(chromPeaks(xdata))[featureDefinitions(xdata)$peakidx[[1L]]]

## We can now pass these to the `chromPeakChromatograms` function with
## parameter `peaks`:
eic_1 <- chromPeakChromatograms(xdata, peaks = peak_ids)

## To plot these into a single plot we need to use the
## `plotChromatogramsOverlay` function:
plotChromatogramsOverlay(eic_1)
}
\seealso{
\code{\link[=filterColumnsKeepTop]{filterColumnsKeepTop()}} to filter the extracted EICs keeping only
the \emph{top n} columns (samples) with the highest intensity.
\code{\link[=chromPeakChromatograms]{chromPeakChromatograms()}} for a function to extract an EIC for each
chromatographic peak.
}
\author{
Johannes Rainer
}
