% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/run_enrichment.R
\name{run_enrichment}
\alias{run_enrichment}
\title{Perform enrichment analysis on selected features from a exposomicset object}
\usage{
run_enrichment(
  exposomicset,
  feature_type = c("degs", "degs_robust", "omics", "factor_features", "degs_cor",
    "omics_cor", "factor_features_cor"),
  score_col = "stability_score",
  score_threshold = NULL,
  variable_map = NULL,
  factor_type = c("common_top_factor_features", "top_factor_features"),
  feature_col = "feature",
  deg_pval_col = "adj.P.Val",
  deg_pval_threshold = 0.05,
  deg_logfc_col = "logFC",
  deg_logfc_threshold = log2(1.5),
  db = c("GO", "KEGG", "Reactome"),
  species = NULL,
  fenr_col = "gene_symbol",
  padj_method = "fdr",
  pval_thresh = 0.1,
  min_set = 3,
  max_set = 800,
  clustering_approach = "diana",
  action = "add"
)
}
\arguments{
\item{exposomicset}{An \code{exposomicset} (a \code{MultiAssayExperiment}
object with metadata) containing omics and metadata.}

\item{feature_type}{Character string indicating the feature source.
One of \code{"degs"}, \code{"degs_robust"},
\code{"omics"}, \code{"factor_features"}, \code{"degs_cor"}, \code{"omics_cor"},
or \code{"factor_features_cor"}.}

\item{score_col}{Column name used for stability score filtering
(only for \code{degs_robust}).}

\item{score_threshold}{Optional numeric threshold for filtering
stability scores. If \code{NULL}, the default
threshold stored in the metadata will be used.}

\item{variable_map}{A data frame with \code{exp_name} and \code{variable} columns,
used when \code{feature_type = "omics"}.}

\item{factor_type}{Character string for selecting factor features:
\code{"common_top_factor_features"} or
\code{"top_factor_features"}.}

\item{feature_col}{The name of the feature column used to extract
gene identifiers.}

\item{deg_pval_col}{Column name for adjusted p-values from DEG analysis.}

\item{deg_pval_threshold}{Threshold to select significant DEGs
(default: 0.05).}

\item{deg_logfc_col}{Column name for log-fold changes from DEG analysis.}

\item{deg_logfc_threshold}{Threshold to select DEGs by absolute logFC
(default: \code{log2(1.5)}).}

\item{db}{Enrichment database to use. One of \code{"GO"},
\code{"KEGG"}, \code{"Reactome"}.}

\item{species}{Species name (required for GO enrichment,
e.g., \code{"Homo sapiens"}). Ignored for other databases.}

\item{fenr_col}{Column name for gene IDs used by \code{fenr}
(e.g., \code{"gene_symbol"}).}

\item{padj_method}{Method for p-value adjustment (default: \code{"fdr"}).}

\item{pval_thresh}{Adjusted p-value threshold for filtering enriched terms
(default: 0.1).}

\item{min_set}{Minimum number of selected genes overlapping an enriched term
(default: 3).}

\item{max_set}{Maximum number of selected genes overlapping an enriched term
(default: 800).}

\item{clustering_approach}{Clustering method for GO term grouping.
Defaults to \code{"diana"}.}

\item{action}{Either \code{"add"} to store results in the object's metadata
or \code{"get"} to return results as a data frame.}
}
\value{
If \code{action = "add"}, returns the modified \code{exposomicset} with
enrichment results added to metadata.
If \code{action = "get"}, returns a \code{data.frame} of enrichment results
with GO term clusters (if applicable).
}
\description{
This function performs enrichment analysis using selected features derived
from differential expression, correlation analysis,
or multi-omics factor features across experiments in an \code{exposomicset}.
It supports multiple enrichment databases (e.g., GO, KEGG, Reactome),
applies FDR correction, and optionally clusters GO terms by Jaccard overlap.
}
\details{
The function identifies selected features based on the chosen
\code{feature_type}, determines the gene universe
for each experiment, and performs enrichment analysis using the
\code{fenr} package. Results are adjusted for
multiple testing and optionally clustered by gene set overlap (for GO terms).

If \code{feature_type} includes correlation-based results
(ending in \verb{_cor}), enrichment is performed for each
exposure category separately.
}
\examples{
# create example data
mae <- make_example_data(
    n_samples = 30,
    return_mae = TRUE
)

# perform differential abundance analysis
mae <- run_differential_abundance(
    exposomicset = mae,
    formula = ~ smoker + sex,
    abundance_col = "counts",
    method = "limma_voom",
    action = "add"
)

# perform enrichment analysis
mae <- run_enrichment(
    exposomicset = mae,
    feature_type = "degs",
    feature_col = "symbol",
    species = "goa_human",
    deg_logfc_threshold = log2(1),
    deg_pval_col = "P.Value",
    deg_pval_threshold = 0.2,
    action = "add"
)

}
