\name{plotfeatureCoverage}
\alias{plotfeatureCoverage}
\title{
Plot feature coverage results
}
\description{
Plots the 3 tabular data types (A-C) generated by the \code{featureCoverage} 
function. It accepts data from single or many features (e.g. CDSs) and
samples (BAM files). The coverage from multiple features will be summarized
using methods such as \code{mean}, while the data from multiple samples will 
be plotted in separate panels.  
}
\usage{
plotfeatureCoverage(covMA, method = mean, scales = "fixed", extendylim=2, 
                    scale_count_val = 10^6)
}
\arguments{
  \item{covMA}{
Object of class \code{data.frame} generated by \code{featureCoverage} function.
}
  \item{method}{
Defines the summary statistics to use when \code{covMA} contains coverage 
data from  multiple features (e.g. transcripts). The default calculates the 
mean coverage for each positon and/or bin of the corresponding coverage 
vectors.
}
  \item{scales}{
Scales setting passed on to the \code{facet_wrap} function of \code{ggplot2}. 
For details see \code{ggplot2::facet_wrap}. The default \code{fixed} assures a
constant scale across all bar plot panels, while \code{free} uses the optimum
scale within each bar plot panel. To evaluate plots in all their details, it
may be necessary to generate two graphics files one for each scaling option. 
}
  \item{extendylim}{
 Allows to extend the upper limit of the y axis when \code{scales=fixed}. Internally, the 
function identifies the maximum value in the data and then multiplies this maximum value 
by the value provided under \code{extendylim}. The default is set to \code{extendylim=2}.
}
  \item{scale_count_val}{
Scales (normalizes) the read counts to a fixed value of aligned reads
in each sample such as counts per million aligned reads (default is 10^6). 
For this calculation the \code{N_total_aligned} values are used that are 
reported in the input \code{data.frame} generated by the upstream 
\code{featureCoverage} function. Assign \code{NULL} to turn off
scaling. 
}
}
\value{
Currently, the function returns ggplot2 bar plot graphics. 
}
\author{
Thomas Girke
}
\seealso{
\code{featureCoverage}
}
\examples{
## Construct SYSargs2 object from param and targets files 
targets <- system.file("extdata", "targets.txt", package="systemPipeR")
dir_path <- system.file("extdata/cwl", package="systemPipeR")
args <- loadWorkflow(targets=targets, wf_file="hisat2/hisat2-mapping-se.cwl", 
                  input_file="hisat2/hisat2-mapping-se.yml", dir_path=dir_path)
args <- renderWF(args, inputvars=c(FileName="_FASTQ_PATH1_", SampleName="_SampleName_"))
args

\dontrun{
## Run alignments
args <- runCommandline(args, dir = FALSE, make_bam = TRUE)
outpaths <- subsetWF(args, slot = "output", subset = 1, index = 1)

## Features from sample data of systemPipeRdata package
library(txdbmaker)
file <- system.file("extdata/annotation", "tair10.gff", package="systemPipeRdata")
txdb <- makeTxDbFromGFF(file=file, format="gff3", organism="Arabidopsis")

## (A) Generate binned coverage for two BAM files and 4 transcripts
grl <- cdsBy(txdb, "tx", use.names=TRUE)
fcov <- featureCoverage(bfl=BamFileList(outpaths[1:2]), grl=grl[1:4], resizereads=NULL, 
                    readlengthrange=NULL, Nbins=20, method=mean, fixedmatrix=FALSE, 
                    resizefeatures=TRUE, upstream=20, downstream=20, 
                    outfile="results/featureCoverage.xls")
plotfeatureCoverage(covMA=fcov, method=mean, scales="fixed", scale_count_val=10^6)

## (B) Coverage matrix upstream and downstream of start/stop codons
fcov <- featureCoverage(bfl=BamFileList(outpaths[1:2]), grl=grl[1:4], resizereads=NULL, 
                    readlengthrange=NULL, Nbins=NULL, method=mean, fixedmatrix=TRUE, 
                    resizefeatures=TRUE, upstream=20, downstream=20, 
                    outfile="results/featureCoverage_UpDown.xls")
plotfeatureCoverage(covMA=fcov, method=mean, scales="fixed", scale_count_val=10^6)

## (C) Combined matrix for both binned and start/stop codon 
fcov <- featureCoverage(bfl=BamFileList(outpaths[1:2]), grl=grl[1:4], resizereads=NULL, 
                    readlengthrange=NULL, Nbins=20, method=mean, fixedmatrix=TRUE, 
                    resizefeatures=TRUE, upstream=20, downstream=20, 
                    outfile="results/test.xls")
plotfeatureCoverage(covMA=fcov, method=mean, scales="fixed", scale_count_val=10^6)

## (D) Rle coverage objects one for each query feature
fcov <- featureCoverage(bfl=BamFileList(outpaths[1:2]), grl=grl[1:4], resizereads=NULL, 
                    readlengthrange=NULL, Nbins=NULL, method=mean, fixedmatrix=FALSE, 
                    resizefeatures=TRUE, upstream=20, downstream=20, 
                    outfile="results/RleCoverage.xls")
}
}

\keyword{ utilities }
