% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/smoothness_metric.R
\name{smoothness_metric}
\alias{smoothness_metric}
\title{Function for smoothness metric}
\usage{
smoothness_metric(spatial_coords, labels, k = 6)
}
\arguments{
\item{spatial_coords}{Numeric matrix containing spatial coordinates of
points, formatted as nrow = number of points, ncol = 2 (assuming x and y
dimensions). For example, `spatial_coords = spatialCoords(spe)` if using a
\code{SpatialExperiment} object.}

\item{labels}{Numeric vector of cluster labels for each point. For example,
`labels <- as.numeric(colData(spe)$label)` if using a
\code{SpatialExperiment} object.}

\item{k}{Number of k nearest neighbors to use in calculation. Default = 6
(e.g. for hexagonal arrangement in 10x Genomics Visium platform).}
}
\value{
Returns a list containing (i) a vector of values at each point (i.e.
  the number of nearest neighbors that are from a different cluster at each
  point) and (ii) the average value across all points.
}
\description{
Function for clustering smoothness evaluation metric
}
\details{
Function to calculate clustering smoothness evaluation metric, defined as the
average number of nearest neighbors per point that are from a different
cluster. This metric can be used to quantify and compare the relative
smoothness of the boundaries of clusters or spatial domains.
}
\examples{
library(STexampleData)
library(scran)
library(scater)

# load data
spe <- Visium_humanDLPFC()
# keep spots over tissue
spe <- spe[, colData(spe)$in_tissue == 1]

# run smoothclust using default parameters
spe <- smoothclust(spe)

# calculate logcounts
spe <- logNormCounts(spe, assay.type = "counts_smooth")

# preprocessing steps for clustering
# remove mitochondrial genes
is_mito <- grepl("(^mt-)", rowData(spe)$gene_name, ignore.case = TRUE)
spe <- spe[!is_mito, ]
# select top highly variable genes (HVGs)
dec <- modelGeneVar(spe)
top_hvgs <- getTopHVGs(dec, prop = 0.1)
spe <- spe[top_hvgs, ]

# dimensionality reduction
set.seed(123)
spe <- runPCA(spe)

# run k-means clustering
set.seed(123)
k <- 5
clus <- kmeans(reducedDim(spe, "PCA"), centers = k)$cluster
colLabels(spe) <- factor(clus)

# calculate smoothness metric
res <- smoothness_metric(spatialCoords(spe), as.numeric(colData(spe)$label))

# results
str(res)
head(res$n_discordant)
res$mean_discordant

}
