#' Infer sex.
#'
#' We established our sex calling based on the CpGs hypermethylated in
#' inactive X (XiH), CpGs hypomethylated in inactive X (XiL).
#'
#' Note genotype abnormalities such as Dnmt genotype,
#' XXY male (Klinefelter's),
#' 45,X female (Turner's) can confuse the model sometimes.
#' This function works on a single sample.
#' 
#' @param betas DNA methylation beta
#' @param platform EPICv2, EPIC, HM450, MM285, etc.
#' @return Inferred sex of sample
#' @examples
#'
#' ## EPICv2 input
#' betas = openSesame(sesameDataGet("EPICv2.8.SigDF")[[1]])
#' inferSex(betas)
#'
#' \dontrun{
#' ## MM285 input
#' betas = openSesame(sesameDataGet("MM285.1.SigDF"))
#' inferSex(betas)
#' 
#' ## EPIC input
#' betas = openSesame(sesameDataGet('EPIC.1.SigDF'))
#' inferSex(betas)
#'
#' ## HM450 input
#' betas = openSesame(sesameDataGet("HM450.10.SigDF")[[1]])
#' inferSex(betas)
#' }
#' 
#' @export
inferSex <- function(betas, platform=NULL) {
  hypoMALE <- c(
    "cg21983484","cg23696472","cg11673471","cg01742836","cg13574945",
    "cg08059778","cg24186901","cg26023405","cg15977272","cg13023833",
    "cg20766178","cg20455959","cg26584339","cg13130271","cg13244998",
    "cg05872808","cg21290550","cg05806018","cg07861180","cg20015269",
    "cg12576145","cg10991108","cg02333283","cg16357225","cg25206026",
    "cg20749341","cg03773146","cg04872051","cg16590821","cg09520212",
    "cg22221554","cg11152253","cg23429746","cg00813156","cg25132467",
    "cg16221895","cg09307104","cg15165114","cg18998000","cg00723973",
    "cg06041068","cg10860619","cg09514431","cg07912337","cg03334316",
    "cg17399684","cg05534333","cg23493872","cg12413138","cg05374090",
    "cg27501007","cg08855111","cg21159768","cg16488754","cg12075609",
    "cg07446674","cg01342901","cg02869694","cg12277627","cg19992190",
    "cg10717149","cg14191108","cg01869765","cg26505478","cg23685102",
    "cg02195366","cg06334238","cg02615131","cg15565409","cg15693668",
    "cg03505772","cg00845806","cg26439324","cg12935118","cg18932686",
    "cg24264679","cg08782677","cg13649400","cg06779802","cg23554546",
    "cg23951868","cg00337921","cg08479532","cg00114625","cg03391801",
    "cg22776211","cg07674503","cg22452543","cg18140045","cg15450782",
    "cg07674075","cg06510592","cg21137943","cg24479484","cg27501723",
    "cg20439892","cg18107314","cg08405463","cg09146364","cg16894263",
    "cg44822048_BC11", "cg48153389_BC11", "cg48114705_BC11", "cg48140091_BC11",
    "cg47832419_BC11", "cg47450117_BC11", "cg47728613_BC11", "cg47583295_TC11",
    "cg47476627_BC11", "cg48109634_BC11", "cg47564226_TC11", "cg47844107_BC11",
    "cg47425903_TC11", "cg47742805_BC21", "cg47855973_BC11", "cg47743423_BC11",
    "cg47906498_TC11", "cg47556267_BC11", "cg47744057_TC21", "cg48176284_BC11",
    "cg48121188_BC11", "cg48065865_BC11", "cg47748343_TC21", "cg47424030_BC21",
    "cg47744023_TC11", "cg47440985_TC11", "cg47583387_BC11", "cg47725474_TC21",
    "cg48024686_TC11", "cg47920249_BC21", "cg48114704_TC11", "cg48148849_BC21",
    "cg47742981_BC11", "cg47743136_BC11", "cg48049840_TC11", "cg48111009_TC21",
    "cg48176352_TC11", "cg47655961_BC11", "cg47856861_BC21", "cg47826283_TC11",
    "cg47901233_TC11", "cg48051845_BC11", "cg47555978_BC21", "cg47634755_BC11",
    "cg48147947_TC11", "cg47503480_BC11", "cg47740318_BC11", "cg48071477_TC11",
    "cg47643035_TC21", "cg47868567_BC11", "cg47655979_TC21", "cg47725912_BC11",
    "cg47564279_BC11", "cg48016415_TC11", "cg47656013_TC21", "cg47873187_TC21",
    "cg47438865_TC11", "cg47906673_BC11", "cg47874829_BC11", "cg47734934_BC21",
    "cg48130287_BC21", "cg47625820_BC21", "cg47505633_TC11", "cg48023062_TC21",
    "cg47744459_TC11", "cg47730002_BC11", "cg47663054_BC11", "cg47742655_BC11",
    "cg48107157_BC11", "cg48148824_TC21", "cg47634666_BC11", "cg47832434_TC11",
    "cg48057717_TC21", "cg48106464_BC11", "cg47748082_TC21", "cg47897499_BC21",
    "cg47889728_TC21", "cg47938210_TC21", "cg48176806_TC11", "cg47740347_BC11",
    "cg48021685_BC21", "cg47612856_BC11", "cg48139201_BC21", "cg48176811_BC11",
    "cg47741292_TC21", "cg47905796_TC21", "cg47643008_TC21", "cg47743984_BC11",
    "cg47795637_BC21", "cg47667056_TC11", "cg48159183_BC21", "cg48164072_TC11",
    "cg48177792_TC21", "cg47743999_TC11", "cg47471551_TC21", "cg47740813_BC21",
    "cg48157924_BC21", "cg47737568_BC11", "cg47724667_BC21", "cg47618975_BC11")

  hyperMALE <- c(
    "cg26359388","cg02540440","cg11049634","cg22874828","cg09182733",
    "cg01123965","cg15822015","cg05130312","cg17072671","cg22655232",
    "cg05695959","cg21010298","cg06143713","cg22759686","cg11143827",
    "cg04303560","cg11717280","cg14372935","cg05533223","cg16405492",
    "cg15765801","cg08156775","cg24183173","cg21797452","cg03161453",
    "cg10474871","cg11516614","cg18813691","cg08614574","cg08456555",
    "cg16440909","cg13326840","cg16822540","cg03801901","cg09039264",
    "cg01383599","cg14931238","cg04071644","cg22208280","cg05559023",
    "cg23317607","cg26327984","cg07801607","cg06870560","cg24156613",
    "cg04101819","cg07422795","cg14261068","cg12622895","cg09192294",
    "cg26695278","cg12653510","cg03554089","cg11166197","cg04032096",
    "cg25047306","cg07818713","cg21258987","cg07981033","cg14492530",
    "cg18157587","cg12030638","cg17498624","cg01816615","cg08723064",
    "cg05193067","cg27167763","cg15521097","cg25456959","cg16576300",
    "cg07318999","cg22417678","cg22671388","cg23644934","cg00267352",
    "cg22223709","cg23698976","cg06780606","cg13920260","cg15861835",
    "cg10039267","cg12454245","cg22067189","cg00150874","cg08401365",
    "cg13781721","cg02931660","cg01316390","cg14746118","cg21294096",
    "cg11871337","cg00408231","cg09641151","cg05226646","cg11291200",
    "cg01109660","cg23607813","cg04624564","cg07452499","cg18123612",
    "cg48211697_TC11", "cg48222828_BC21", "cg48218650_BC21", "cg48219904_BC11",
    "cg48222534_TC11", "cg48214483_TC21", "cg48222923_TC11", "cg48217358_TC12",
    "cg48217547_BC11", "cg48215035_TC11", "cg48217358_TC11", "cg48215051_TC21",
    "cg48218172_BC21", "cg48218223_TC11", "cg48296014_TC11", "cg48218620_TC11",
    "cg48213060_TC11", "cg48244014_TC11", "cg48215477_BC11", "cg48217390_BC11",
    "cg48272545_BC11", "cg48222620_BC11", "cg48309797_TC11", "cg48212920_TC11",
    "cg48218860_TC11", "cg48216374_TC11", "cg48215185_BC21", "cg48213802_BC11",
    "cg48222396_TC11", "cg48214010_BC11", "cg48222395_BC11", "cg48218465_TC21",
    "cg48215216_BC11", "cg48216938_BC11", "cg48219858_BC21", "cg48214243_BC11",
    "cg48223281_TC21", "cg48214292_BC21", "cg32022449_BC11", "cg48215159_TC21",
    "cg48222049_BC11", "cg48246403_BC21", "cg48214455_BC11", "cg48216569_TC11",
    "cg48214177_BC21", "cg48246617_BC11", "cg48301218_BC11", "cg48214011_BC11",
    "cg48215297_BC21", "cg48217555_BC21", "cg48213764_TC21", "cg48222839_TC11",
    "cg48217418_TC21", "cg48216934_BC21", "cg48250058_BC11", "cg48219493_TC21",
    "cg48222602_TC21", "cg48217485_BC12", "cg48218187_TC11", "cg48222171_TC11",
    "cg48217401_BC21", "cg48218225_BC11", "cg48222795_BC11", "cg48224019_TC11",
    "cg48217672_BC11", "cg48217626_TC21", "cg48213632_BC21", "cg48216281_TC21",
    "cg48218341_BC21", "cg48222701_BC11", "cg48218522_TC11", "cg48217489_BC11",
    "cg48212144_TC21", "cg48219215_TC21", "cg48218176_BC11", "cg48223101_BC11",
    "cg48222143_TC11", "cg48218124_BC21", "cg48218975_BC11", "cg48217449_TC21",
    "cg48222478_BC21", "cg48216323_BC21", "cg48217683_BC11", "cg48215310_TC21",
    "cg48226387_BC11", "cg48218807_BC11", "cg48213481_BC11", "cg48224372_BC11",
    "cg48217446_BC21", "cg48222402_TC11", "cg48222222_TC11", "cg48215306_BC21",
    "cg48219235_BC21", "cg48221203_TC11", "cg48216903_BC21", "cg48218631_BC21",
    "cg48220121_TC11", "cg48215553_TC11", "cg48217396_TC11", "cg48224236_BC21")

  platform <- sesameData_check_platform(platform, names(betas))
  if (platform != "MM285") {
      betas <- mLiftOver(betas, "HM450")
  }
  vals <- mean(betas[hyperMALE], na.rm = TRUE) - betas[hypoMALE]
  dd <- density(na.omit(vals))
  if (dd$x[which.max(dd$y)] > 0.4) {
    "MALE"
  } else {
    "FEMALE"
  }
}
