% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/countRandomBarcodes.R
\name{countRandomBarcodes}
\alias{countRandomBarcodes}
\alias{matrixOfRandomBarcodes}
\title{Count random barcodes}
\usage{
countRandomBarcodes(
  fastq,
  template,
  substitutions = 0,
  find.best = FALSE,
  strand = c("both", "original", "reverse"),
  num.threads = 1
)

matrixOfRandomBarcodes(
  files,
  ...,
  withDimnames = TRUE,
  BPPARAM = SerialParam()
)
}
\arguments{
\item{fastq}{String containing the path to a FASTQ file containing single-end data.}

\item{template}{String containing the template for the barcode structure.
See \code{\link{parseBarcodeTemplate}} for more details.}

\item{substitutions}{Integer scalar specifying the maximum number of substitutions when considering a match.}

\item{find.best}{Logical scalar indicating whether to search each read for the best match.
Defaults to stopping at the first match.}

\item{strand}{String specifying which strand of the read to search.}

\item{num.threads}{Integer scalar specifying the number of threads to use to process a single file.}

\item{files}{A character vector of paths to FASTQ files.}

\item{...}{Further arguments to pass to \code{countSingleBarcodes}.}

\item{withDimnames}{A logical scalar indicating whether the rows and columns should be named.}

\item{BPPARAM}{A \linkS4class{BiocParallelParam} object specifying how parallelization is to be performed across files.}
}
\value{
\code{countRandomBarcodes} will return a \linkS4class{DataFrame} containing:
\itemize{
\item \code{sequences}, a character vector containing the sequences of the random barcodes in the variable region.
\item \code{counts}, an integer vector containing the frequency of each barcode.
}
The metadata contains \code{nreads}, an integer scalar containing the total number of reads in \code{fastq}.

\code{matrixOfRandomBarcodes} will return a \linkS4class{SummarizedExperiment} object containing:
\itemize{
\item An integer matrix named \code{"counts"}, where each column is the output of \code{countRandomBarcodes} for each file in \code{files}.
\item Row metadata containing a character vector \code{sequences}, the sequence of the variable region of each barcode for each row.
\item Column metadata containing a character vector \code{files}, the path to each file;
an integer vector \code{nreads}, containing the total number of reads in each file;
and \code{nmapped}, containing the number of reads assigned to a barcode in the output count matrix.
}
If \code{withDimnames=TRUE}, row names are set to \code{sequences} while column names are \code{basename(files)}.
}
\description{
Count the frequency of random barcodes in a FASTQ file containing data for a single-end sequencing screen.
This differs from \code{\link{countSingleBarcodes}} in that the barcode is completely random rather than being drawn from a known pool of sequences.
}
\details{
If \code{strand="both"}, the original read sequence will be searched first.
If no match is found, the sequence is reverse-complemented and searched again.
Other settings of \code{strand} will only search one or the other sequence.
The most appropriate choice depends on both the sequencing protocol and the design (i.e., position and length) of the barcode.

We can handle sequencing errors by setting \code{substitutions} to a value greater than zero.
This will consider substitutions in both the variable region as well as the constant flanking regions.

By default, the function will stop at the first match that satisfies the requirements above.
If \code{find.best=TRUE}, we will instead try to find the best match with the fewest mismatches.
If there are multiple matches with the same number of mismatches, the read is discarded to avoid problems with ambiguity.
}
\examples{
# Creating an example dataset.
N <- 1000
randomized <- lapply(1:N, function(i) {
    paste(sample(c("A", "C", "G", "T"), 8, replace=TRUE), collapse="")
})
barcodes <- sprintf("CAGCTACGTACG\%sCCAGCTCGATCG", randomized)
names(barcodes) <- seq_len(N)

library(Biostrings)
tmp <- tempfile(fileext=".fastq")
writeXStringSet(DNAStringSet(barcodes), filepath=tmp, format="fastq")

# Counting the sequences:
countRandomBarcodes(tmp, template="CAGCTACGTACGNNNNNNNNCCAGCTCGATCG")

matrixOfRandomBarcodes(c(tmp, tmp), template="CAGCTACGTACGNNNNNNNNCCAGCTCGATCG")
}
\author{
Aaron Lun
}
