% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/F_reconsi.R
\name{reconsi}
\alias{reconsi}
\title{Perform simultaneous inference through
collapsed resampling null distributions}
\usage{
reconsi(
  Y,
  x = NULL,
  B = 1000L,
  test = "wilcox.test",
  argList = list(),
  distFun = "pnorm",
  zValues = TRUE,
  testPargs = list(),
  z0Quant = 0.25,
  gridsize = 801L,
  maxIter = 100L,
  tol = 1e-06,
  zVals = NULL,
  center = FALSE,
  replace = is.null(x),
  assumeNormal = TRUE,
  estP0args = list(z0quantRange = seq(0.05, 0.45, 0.0125), smooth.df = 3, evalVal = 0.05),
  resamZvals = FALSE,
  smoothObs = TRUE,
  scale = FALSE,
  tieBreakRan = FALSE,
  pi0 = NULL,
  resamAssumeNormal = TRUE
)
}
\arguments{
\item{Y}{the matrix of sequencing counts}

\item{x}{a grouping factor. If provided, this grouping factor is permuted.
Otherwise a bootstrap procedure is performed}

\item{B}{the number of resampling instances}

\item{test}{Character string, giving the name of the function
to test for differential absolute abundance.
Must accept the formula interface. Features with tests resulting in
observed NA test statistics will be discarded}

\item{argList}{A list of arguments, passed on to the testing function}

\item{distFun}{the distribution function of the test statistic, or its name.
Must at least accept an argument named 'q', 'p' and 'x' respectively.}

\item{zValues}{A boolean, should test statistics be converted to z-values.
See details}

\item{testPargs}{A list of arguments passed on to distFun}

\item{z0Quant}{A vector of length 2 of quantiles of the null distribution,
in between which only null values are expected}

\item{gridsize}{The number of bins for the kernel density estimates}

\item{maxIter}{An integer, the maximum number of iterations in the estimation
of the null distribution}

\item{tol}{The tolerance for the infinity norm of the central borders
in the iterative procedure}

\item{zVals}{An optional list of observed (statObs) and
resampling (statsPerm) z-values. If supplied, the calculation
   of the observed and resampling test statistics is skipped
   and the algorithm proceeds with calculation
   of the consensus null distribution}

\item{center}{A boolean, should observations be centered
in each group prior to permuations? See details.}

\item{replace}{A boolean. Should resampling occur with replacement (boostrap)
or without replacement (permutation)}

\item{assumeNormal}{A boolean, should normality be assumed for the null distribution?}

\item{estP0args}{A list of arguments passed on to the estP0 function}

\item{resamZvals}{A boolean, should resampling rather than theoretical null
distributions be used?}

\item{smoothObs}{A boolean, should the fitted rather than estimated observed
distribution be used in the Fdr calculation?}

\item{scale}{a boolean, should data be scaled prior to resampling}

\item{tieBreakRan}{A boolean, should ties of resampling test statistics
be broken randomly? If not, midranks are used}

\item{pi0}{A known fraction of true null hypotheses. If provided,
the fraction of true null hypotheses will not be estimated.
Mainly for oracle purposes.}

\item{resamAssumeNormal}{A boolean, should normality be assumed for resampling dists}
}
\value{
A list with entries
\item{statsPerm}{Resampling Z-values}
\item{statObs}{Observed Z-values}
\item{distFun}{Density, distribution and
quantile function as given}
\item{testPargs}{Same as given}
\item{zValues}{A boolean, were z-values used?}
\item{resamZvals}{A boolean, were the resampling null distribution used?}
\item{cdfValObs}{Cumulative distribution function evaluation
of observed test statistics}
\item{p0estimated}{A boolean, was the fraction of true null hypotheses
estimated from the data?}
\item{Fdr,fdr}{Estimates of tail-area and local false discovery rates}
\item{p0}{Estimated or supplied fraction of true null hypotheses}
\item{iter}{Number of iterations executed}
\item{fitAll}{Mean and standard deviation estimated collapsed null}
\item{PermDensFits}{Mean and standard deviations of resamples}
\item{convergence}{A boolean, did the iterative algorithm converge?}
\item{zSeq}{Basis for the evaluation of the densities}
\item{weights}{weights of the resampling distributions}
\item{zValsDensObs}{Estimated overall densities, evaluated in zSeq}
}
\description{
Perform simultaneous inference through
collapsed resampling null distributions
}
\details{
Efron (2007) centers the observations in each group prior
 to permutation. As permutations will remove any genuine group differences
  anyway, we skip this step by default. If zValues = FALSE,
  the density is fitted on the original test statistics rather than converted
  to z-values. This unlocks the procedure for test statistics with unknown
  distributions, but may be numerically less stable.
}
\note{
Ideally, it would be better to only use unique resamples, to avoid
unnecesarry replicated calculations of the same test statistics. Yet this
issue is almost alwyas ignored in practice; as the sample size grows it also
becomes irrelevant. Notice also that this would require to place weights in
case of the bootstrap, as some bootstrap samples are more likely than others.
}
\examples{
#Important notice: low number of resamples B necessary to keep
# computation time down, but not recommended. Pray set B at 200 or higher.
p = 60; n = 20; B = 5e1
x = rep(c(0,1), each = n/2)
mat = cbind(
matrix(rnorm(n*p/10, mean = 5+x), n, p/10), #DA
matrix(rnorm(n*p*9/10, mean = 5), n, p*9/10) #Non DA
)
fdrRes = reconsi(mat, x, B = B)
fdrRes$p0
#Indeed close to 0.9
estFdr = fdrRes$Fdr
#The estimated tail-area false discovery rates.

#With another type of test. Need to supply quantile function in this case
fdrResLm = reconsi(mat, x, B = B,
test = function(x, y){
fit = lm(y~x)
c(summary(fit)$coef["x","t value"], fit$df.residual)},
distFun = function(q){pt(q = q[1], df = q[2])})

#With a test statistic without known null distribution(for small samples)
fdrResKruskal = reconsi(mat, x, B = B,
test = function(x, y){
kruskal.test(y~x)$statistic}, zValues = FALSE)

#Provide an additional covariate through the 'argList' argument
z = rpois(n , lambda = 2)
fdrResLmZ = reconsi(mat, x, B = B,
test = function(x, y, z){
fit = lm(y~x+z)
c(summary(fit)$coef["x","t value"], fit$df.residual)},
distFun = function(q){pt(q = q[1], df = q[2])},
argList = list(z = z))

#When nog grouping variable is provided, a bootstrap is performed
matBoot = cbind(
matrix(rnorm(n*p/10, mean = 1), n, p/10), #DA
matrix(rnorm(n*p*9/10, mean = 0), n, p*9/10) #Non DA
)
fdrResBoot = reconsi(matBoot, B = B,
test = function(y, x){testRes = t.test(y, mu = 0, var.equal = TRUE);
c(testRes$statistic, testRes$parameter)},
distFun = function(q){pt(q = q[1], df = q[2])},
center = TRUE, replace = TRUE)
}
