% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/randRot.R
\name{rotateStat}
\alias{rotateStat}
\title{Generate data rotations and calculate statistics on it}
\usage{
rotateStat(
  initialised.obj,
  R = 10,
  statistic,
  ...,
  parallel = FALSE,
  BPPARAM = BiocParallel::bpparam()
)
}
\arguments{
\item{initialised.obj}{An initialised random rotation object as returned by \code{\link[randRotation:initRandrot]{initRandrot}} and \code{\link[randRotation:initRandrot]{initBatchRandrot}}.}

\item{R}{The number of resamples/rotations. Single \code{numeric} larger than 1.}

\item{statistic}{A function which takes a data matrix (same dimensions as \code{Y} - see also \code{\link[randRotation:initRandrot]{initRandrot}}) as first argument and returns a statistic of interest. Any further arguments are passed to it with the \code{...} argument.
We highly recommend using pivotal quantities as \code{statistic} if possible (see also \code{Details} in \code{\link[randRotation:pFdr]{pFdr}}).
Note that \code{\link[randRotation:pFdr]{pFdr}} considers larger values of statistics as more significant, so one-tailed tests may require reversal of the sign and two-tailed tests may require taking absolute values, see \code{Examples}.
The results of \code{statistic} for each resample are finally combined with \code{as.matrix} and \code{cbind}, so ensure that \code{statistic} returns either a vector or a matrix.
Results with multiple columns are possible and handled adequately in subsequent functions (e.g. \code{\link[randRotation:pFdr]{pFdr}}). Note that \code{statistic} must not necessarily be of the same length as \code{nrow(Y)}, but can also be e.g. a summary statistic of genes (like in gene set testing).}

\item{...}{Further named arguments for \code{statistic} which are passed unchanged each time it is called.
Avoid partial matching to arguments of \code{rotateStat}. See also the \code{Examples}.}

\item{parallel}{\code{logical} if parallel computation should be performed, see details for use of parallel computing.}

\item{BPPARAM}{An optional \code{\link[BiocParallel:BiocParallelParam-class]{BiocParallelParam}} instance, see documentation of \code{BiocParallel} package of Bioconductor.}
}
\value{
An object of class \code{\link[randRotation:rotateStat-class]{rotateStat}}.
}
\description{
This function generates rotations of data and calculates the provided \code{statistic} on each rotation and the non-rotated (original) data.
This is the central function of the package.
}
\details{
The function takes an initialised randrot object
(\code{\link[randRotation:initRandrot]{initRandrot}}) and a function that
calculates a statistic on the data. The statistic function thereby takes the
a matrix \code{Y} as first argument. Any further arguments are passed to it
by \code{...}.

Together with \code{\link[randRotation:pFdr]{pFdr}}, this function implements
the \code{workflow} described in \insertCite{Hettegger2021}{randRotation}.

Be aware that only data is rotated (see also
\code{\link[randRotation:randrot]{randrot}}), so any additional information
including \code{weights}, \code{X} etc. need to be provided to
\code{statistic}. See also package vignette and \code{Examples}.

Parallel processing is implemented with the BiocParallel package of Bioconductor.
The default argument \code{\link[BiocParallel:register]{BiocParallel::bpparam()}} for \code{BPPARAM}
returns the registered default backend.
See package documentation for further information and usage options.
If \code{parallel = TRUE} the function calls in \code{statistic} need to be called explicitly
with package name and "::". So e.g. calling \code{lmFit} from the
\code{limma} package is done with \code{limma::lmFit(...)}, see also the
examples in the package vignette.
}
\examples{
#set.seed(0)

# Dataframe of phenotype data (sample information)
# We simulate 2 sample classes processed in 3 batches
pdata <- data.frame(batch = rep(1:3, c(10,10,10)),
                   phenotype = rep(c("Control", "Cancer"), c(5,5)))
features <- 100

# Matrix with random gene expression data
edata <- matrix(rnorm(features * nrow(pdata)), features)
rownames(edata) <- paste("feature", 1:nrow(edata))

mod1 <- model.matrix(~phenotype, pdata)

# Initialisation of the random rotation class
init1 <- initBatchRandrot(Y = edata, X = mod1, coef.h = 2, batch = pdata$batch)
init1

# Definition of the batch effect correction procedure with subsequent calculation
# of two-sided test statistics
statistic <- function(., batch, mod, coef){

  # The "capture.output" and "suppressMessages" simply suppress any output
  capture.output(suppressMessages(
    Y.tmp <- sva::ComBat(., batch = batch, mod)
  ))

  fit1 <- lm.fit(mod, t(Y.tmp))
  abs(coef(fit1)[coef,])
}

# We calculate test statistics for the second coefficient

res1 <- rotateStat(initialised.obj = init1,
                    R = 10,
                    statistic = statistic,
                    batch = pdata$batch, mod = mod1, coef = 2)

hist(pFdr(res1))
}
\references{
\insertAllCited{}
}
\author{
Peter Hettegger
}
