% Generated by roxygen2 (4.0.0): do not edit by hand
\name{makeCalls}
\alias{makeCalls}
\title{Make antibody binding positivity calls}
\usage{
makeCalls(peptideSet, cutoff = 1.2, method = "absolute", freq = TRUE,
  group = NULL, verbose = FALSE)
}
\arguments{
\item{peptideSet}{A \code{peptideSet} object. The peptides, after normalization
and possibly data smoothing.}

\item{cutoff}{A \code{numeric}. If FDR, the FDR threshold. Otherwise, a cutoff
for the background corrected intensities.}

\item{method}{A \code{character}. The method used to make positivity calls.
"absolute" and "FDR" are available. See details below.}

\item{freq}{A \code{logical}. If set to TRUE, return the percentage of slides
calling a peptide positive. Otherwise, return a \code{logical} indicating binding
events.}

\item{group}{A \code{character}. Only used when freq is set to TRUE. A character indicating
a variable by which to group slides. If non-null the percentage is calculated by group.}

\item{verbose}{A \code{logical}. If set to TRUE, progress information will be displayed.}
}
\value{
If freq = TRUE, a \code{numeric} \code{matrix} with peptides as rows and
groups as columns where the values are the frequency of response in the group. If
freq = FALSE, a \code{logical} \code{matrix} indicating binding events for each
peptide in each subject.
}
\description{
After normalization and data smoothing, this last step makes the call for each
peptide of the peptideSet after baseline correcting the peptide intenstities.
}
\details{
This function requires specific variables ptid and visit in pData(peptideSet).
The variable \code{ptid} should indicate subjects, and the variable \code{visit}
should be a factor with levels pre and post.

If slides are paired for subjects, intensities corresponding to post-visit are
substracted from pre. If slides are not paired, slides with pre have intensities
averaged by peptides, and averaged peptide intensities are subtracted from slides
that have entry post. Calls are made on these baseline corrected intensities.

When method = FDR, a left-tail method is used to generate a threshold controlling
the False Discovery Rate at level \code{cutoff}. When method = absolute, Intensities
exceeding the threshold are labelled as positive.

When freq = TRUE a group variable may be specified. The argument group indicates
the name of a variable in pData(peptideSet) by which positive calls should be grouped.
The call frequency for each peptide is calculated within groups.
}
\examples{
## This example curated from the vignette -- please see vignette("pepStat")
## for more information
if (require("pepDat")) {

  ## Get example GPR files + associated mapping file
  dirToParse <- system.file("extdata/gpr_samples", package = "pepDat")
  mapFile <- system.file("extdata/mapping.csv", package = "pepDat")

  ## Make a peptide set
  pSet <- makePeptideSet(files = NULL, path = dirToParse,
                         mapping.file = mapFile, log=TRUE)

  ## Plot array images -- useful for quality control
  plotArrayImage(pSet, array.index = 1)
  plotArrayResiduals(pSet, array.index = 1, smooth = TRUE)

  ## Summarize peptides, using pep_hxb2 as the position database
  data(pep_hxb2)
  psSet <- summarizePeptides(pSet, summary = "mean", position = pep_hxb2)

  ## Normalize the peptide set
  pnSet <- normalizeArray(psSet)

  ## Smooth
  psmSet <- slidingMean(pnSet, width = 9)

  ## Make calls
  calls <- makeCalls(psmSet, freq = TRUE, group = "treatment",
                     cutoff = .1, method = "FDR", verbose = TRUE)

  ## Produce a summary of the results
  summary <- restab(psmSet, calls)

}
}
\author{
Greg Imholte
}

