% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/multiModalMNN.R
\name{multiModalMNN}
\alias{multiModalMNN}
\title{Multi-modal batch correction with MNNs}
\usage{
multiModalMNN(
  ...,
  batch = NULL,
  which = NULL,
  rescale.k = 50,
  common.args = list(),
  main.args = list(),
  alt.args = list(),
  mnn.args = list(),
  BNPARAM = KmknnParam(),
  BPPARAM = SerialParam()
)
}
\arguments{
\item{...}{One or more \linkS4class{SingleCellExperiment} objects,
containing a shared set of alternative Experiments corresponding to different data modalities.
Alternatively, one or more lists of such objects.}

\item{batch}{Factor specifying the batch to which each cell belongs, when \code{...} contains only one SingleCellExperiment object.
Otherwise, each object in \code{...} is assumed to contain cells from a single batch.}

\item{which}{Character vector containing the names of the alternative Experiments to use for correction.
Defaults to the names of all alternative Experiments that are present in every object of \code{...}.}

\item{rescale.k}{Integer scalar specifying the number of neighbors to use in \code{\link{rescaleByNeighbors}}.}

\item{common.args}{Named list of further arguments to control the PCA for all modalities.}

\item{main.args}{Named list of further arguments to control the PCA for the main Experiments.
Overrides any arguments of the same name in \code{common.args}.}

\item{alt.args}{Named list of named lists of further arguments to control the PCA for each alternative Experiment specified in \code{which}.
This should be a list where each entry is named after any alternative Experiment and contains an internal list of named arguments;
these override any settings in \code{common.args} in the PCA for the corresponding modality.}

\item{mnn.args}{Further arguments to pass to \code{\link{reducedMNN}}, controlling the MNN correction.}

\item{BNPARAM}{A \linkS4class{BiocNeighborParam} object specifying how the nearest neighbor searches should be performed.}

\item{BPPARAM}{A \linkS4class{BiocParallelParam} object specifying how parallelization should be performed.}
}
\value{
A \linkS4class{SingleCellExperiment} of the same structure as that returned by \code{\link{fastMNN}},
i.e., with a \code{corrected} entry of corrected low-dimensional coordinates and a \code{reconstructed} assay of corrected expression values.
In addition, the \code{\link{altExps}} entries contain corrected values for each data modality used in the correction.
}
\description{
Perform MNN correction on multi-modal data, based on a generalization of \code{\link{fastMNN}} to multiple feature sets.
}
\details{
This function implements a multi-modal MNN correction for SingleCellExperiment inputs where each main and alternative Experiment corresponds to one modality.
We perform a PCA within each modality with \code{\link{multiBatchPCA}},
rescale the PCs to be of a comparable scale with \code{\link{rescaleByNeighbors}},
and finally correct in low-dimensional space with \code{\link{reducedMNN}}.
Corrected expression values for each modality are then recovered in the same manner as described for \code{\link{fastMNN}}.

Modality-specific arguments can be passed to the PCA step via the \code{common.args}, \code{main.args} and \code{alt.args} arguments.
These mirror the corresponding arguments in \code{\link{applyMultiSCE}} - see the documentation for that function for more details.
Additional arguments for the MNN step can be passed via \code{mnn.args}.
Note that \code{batch} is used across all steps and must be specified as its own argument in the \code{multiModalMNN} function signature.

Most arguments in \code{\link{multiBatchPCA}} can be specified in \code{common.args}, \code{main.args} or each entry of \code{alt.args}.
This includes passing \code{d=NA} to turn off the PCA or \code{subset.row} to only use a subset of features for the PCA.
Additionally, the following arguments are supported:
\itemize{
\item By default, a cosine-normalization is performed prior to the PCA for each modality.
This can be disabled by passing \code{cos.norm=FALSE} to \code{common.args}, \code{main.args} or each entry of \code{alt.args}.
\item Setting \code{correct.all} will reported corrected expression values for all features even when \code{subset.row} is specified.
This can be used in \code{common.args}, \code{main.args} or each entry of \code{alt.args}.
}

Note that the function will look for \code{assay.type="logcounts"} by default in each entry of \code{...}.
Users should perform log-normalization prior to calling \code{multiModalMNN}, most typically with \code{\link{multiBatchNorm}} - see Examples.
}
\examples{
# Mocking up a gene expression + ADT dataset:
library(scater)
exprs_sce <- mockSCE()
adt_sce <- mockSCE(ngenes=20) 
altExp(exprs_sce, "ADT") <- adt_sce

# Pretend we have three batches for the sake of demonstration:
batch <- sample(1:3, ncol(exprs_sce), replace=TRUE)

# Normalizing first with batchelor::multiBatchNorm:
library(batchelor)
exprs_sce <- applyMultiSCE(exprs_sce, batch=batch, FUN=multiBatchNorm)

# and perform batch correction:
corrected <- multiModalMNN(exprs_sce, batch=batch, which="ADT")

# Pass arguments to, e.g., use a subset of features for the RNA,
# turn off the PCA for the ADTs:
corrected2 <- multiModalMNN(exprs_sce, batch=batch, which="ADT",
    main.args=list(subset.row=1:500), 
    alt.args=list(ADT=list(d=NA)))

}
\seealso{
\code{\link{fastMNN}}, for MNN correction within a single modality.

\code{\link{multiBatchPCA}}, to perform a batch-aware PCA within each modality.

\code{\link{applyMultiSCE}}, which inspired this interface for Experiment-specific arguments.
}
\author{
Aaron Lun
}
