% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/groupedStouffer.R
\name{groupedStouffer}
\alias{groupedStouffer}
\title{Combine grouped p-values with Stouffer's Z-score method}
\usage{
groupedStouffer(p.values, grouping, weights = NULL, log.p = FALSE)
}
\arguments{
\item{p.values}{A numeric vector containing p-values for individual tests.}

\item{grouping}{A vector or factor of length equal to \code{p.values}, specifying the group to which each test is assigned.

Alternatively, an \link{rle} object where each run corresponds to a group and specifies the entries of \code{p.values} belonging to that group.
This assumes that \code{p.values} is ordered such that all entries in the same group are adjacent to each other.}

\item{weights}{A numeric vector of length equal to \code{p.values}, containing a positive weight for each test.
Alternatively \code{NULL}, in which case equal weights are assigned to all tests.}

\item{log.p}{Logical scalar indicating whether the p-values in \code{p.values} are log-transformed.}
}
\value{
A list containing:
\itemize{
\item \code{p.value}, a named numeric vector of length equal to the number of unique levels in \code{grouping}.
This contains the combined p-value for each group, log-transformed if \code{log.p=TRUE}.
Each entry is named according to the group.
\item \code{representative}, a named integer vector containing the indices of the representative tests for each group.
Each index refers to an entry of \code{p.values} and is named according to its group.
\item \code{influential}, a logical vector of length equal to \code{p.values}.
Entries are \code{TRUE} for any p-value that is deemed \dQuote{influential} to the final combined p-value for its group.
}
}
\description{
Combine p-values from grouped tests with Stouffer's Z-score method.
Groups are defined according to unique levels of a grouping factor.
}
\details{
The joint null hypothesis for each group is that all of the individual null hypotheses are true.
Stouffer's method combines information from all individual nulls to determine if the joint null should be rejected.
This serves as a compromise between Fisher's method (sensitive to the smallest p-value) and Pearson's method (sensitive to the largest p-value).

Stouffer's method is only applicable for independent tests.
Weights are supported by scaling the contribution of each individual null to the summed Z-score.
In this manner, more highly weighted tests will have a greater effect on the final combined p-value.

The representative test for each group is defined as the test with the most negative weighted Z-score, 
as this has the greatest effect on the combined p-value when the joint null is rejected. 
All tests for each group are considered to be influential as increasing any of them (e.g., to unity) would result in a larger combined p-value.

When a group contains both zero and unity p-values, we compare the sum of weights for all zero p-values and all unity p-values.
If the former is larger, the combined p-value is set to zero; if the latter, to unity.
If they are equal, we pretend that the two sets of p-values \dQuote{cancel out} and contribute nothing to the summed Z-score.
This is not entirely rigorous but provides reasonable output in the presence of such boundary values.
}
\examples{
p1 <- rbeta(100, 0.8, 1)
g <- sample(10, length(p1), replace=TRUE)

# Standard application:
out <- groupedStouffer(p1, g)
str(out)

# With weights:
out <- groupedStouffer(p1, g, weights=rexp(length(p1)))
str(out)

# With log p-values. 
out <- groupedStouffer(log(p1), g, log.p=TRUE)
str(out)

}
\references{
Stouffer SA et al. (1949).
\emph{The American Soldier, Vol. 1 - Adjustment during Army Life}.
Princeton University Press (Princeton).

Whitlock MC (2005).
Combining probability from independent tests: the weighted Z-method is superior to Fisher's approach.
\emph{J. Evol. Biol.} 18, 5:1368-73.
}
\seealso{
\code{\link{parallelStouffer}}, for a version that operates on parallel vectors of p-values.

\code{\link{groupedFisher}} and \code{\link{groupedPearson}}, for different approaches to testing a joint null of independent hypotheses.
}
\author{
Aaron Lun
}
