\name{filt1}

\alias{filt1}

\alias{filt1_ma}
\alias{filt1_conv}
\alias{filt1_gauss}
\alias{filt1_bi}
\alias{filt1_adapt}
\alias{filt1_diff}
\alias{filt1_guide}
\alias{filt1_pag}
\alias{filt1_sg}

\title{Smoothing Filters in 1D}

\description{
    Smooth a uniformly sampled 1D signal.
}

\usage{
# Moving average filter
filt1_ma(x, width = 5L)

# Linear convolution filter
filt1_conv(x, weights)

# Gaussian filter
filt1_gauss(x, width = 5L, sd = (width \%/\% 2) / 2)

# Bilateral filter
filt1_bi(x, width = 5L, sddist = (width \%/\% 2) / 2,
    sdrange = mad(x, na.rm = TRUE))

# Bilateral filter with adaptive parameters
filt1_adapt(x, width = 5L, spar = 1)

# Nonlinear diffusion
filt1_diff(x, niter = 3L, kappa = 50,
    rate = 0.25, method = 1L)

# Guided filter
filt1_guide(x, width = 5L, guide = x,
    sdreg = mad(x, na.rm = TRUE))

# Peak-aware guided filter
filt1_pag(x, width = 5L, guide = NULL,
    sdreg = mad(x, na.rm = TRUE), ftol = 1/10)

# Savitzky-Golay filter
filt1_sg(x, width = 5L, order = min(3L, width - 2L),
    deriv = 0, delta = 1)
}

\arguments{
	\item{x}{A numeric vector.}

    \item{width}{The width of the smoothing window in number of samples. Must be positive. Must be odd.}

    \item{weights}{The weights of the linear convolution kernel. Length must be odd.}

    \item{sd, sddist}{The spatial parameter for kernel-based filters. This controls the strength of smoothing for samples farther from the center of the smoothing window.}

    \item{sdrange}{The range parameter for kernel-based filters. This controls the strength of the smoothing for samples with signal values very different from the center of the smoothing window.}

    \item{spar}{The strength of the smoothing when calculating the adaptive bilateral filtering parameters. The larger the number, the stronger the smoothing. Must be positive.}

    \item{kappa}{The constant for the conduction coefficient for nonlinear diffusion. Must be positive.}

    \item{rate}{The rate of diffusion. Must be between 0 and 0.25 for stability.}

    \item{method}{The diffusivity method, where \code{1} and \code{2} correspond to the two diffusivity functions proposed by Perona and Malik (1990). For \code{1}, this is \code{exp(-(|grad x|/K)^2)}, and \code{2} is \code{1/(1+(|grad x|/K)^2)}. An additional method \code{3} implements the peak-aware weighting \code{1/(1+(|x|/K)^2)}, which does \emph{not} use the gradient, but is used to create the guidance signal for peak-aware guided filtering.}

    \item{niter}{The number of iterations for nonlinear diffusion. Must be positive.}

    \item{guide}{The guide signal for guided filtering. This is the signal used to determine the degree of filtering for different regions of the sample. By default, it is the same as the signal to be smoothed.}

    \item{sdreg}{The regularization parameter for guided filtering. This is analagous to the range parameter for kernel-based filters. Signal regions with variance much smaller than this value are smoothed, while signal regions with varaince much larger than this value are preserved.}

	\item{ftol}{Specifies how large the signal value must be before it is considered a peak, expressed as a fraction of the maximum value in the signal.}

    \item{order}{The polynomial order for the Savitzky-Golay filter coefficients.}

    \item{deriv}{The order of the derivative for the Savitzky-Golay filter coefficients.}

    \item{delta}{The sample spacing for the Savitzky-Golay filter. Only used if \code{deriv > 0}.}
}

\details{
    \code{filt1_ma()} performs mean filtering in O(n) time. This is fast and especially useful for calculating other filters that can be constructed as a combination of mean filters.

    \code{filt1_gauss()} performs Gaussian filtering.

    \code{filt1_bi()} and \code{filt1_adapt()} perform edge-preserving bilateral filtering. The latter calculates the kernel parameters adapatively based on the local signal, using a strategy adapted from Joseph and Periyasamy (2018).

    \code{filt1_diff()} performs the nonlinear diffusion filtering of Perona and Malik (1990). Rather than relying on a filter width, it progressively diffuses (smooths) the signal over multiple iterations. More iterations will result in a smoother image.

    \code{filt1_guide()} performs edge-preserving guided filtering. Guided filtering uses a local linear model based on the structure of a so-called "guidance signal". By default, the guidance signal is often the same as the input signal. Guided filtering performs similarly to bilateral filtering, but is often faster (though with more memory use), as it is implemented as a combination of mean filters.

    \code{filt1_pag()} performs peak-aware guided filtering using a regularization parameter that focuses on preserving peaks rather than edges, using a strategy adapted from Liu and He (2022). By default, the guidance signal is generated by smoothing the input signal with nonlinear diffusion.

    \code{filt1_sg()} performs traditional Savitzky-Golay filtering, which uses a local least-squares polynomial approximation to perform the smoothing. It reduces noise while attempting to retain the peak shape and height.
}

\value{
    A numeric vector the same length as \code{x} with the smoothed result.
}

\author{Kylie A. Bemis}

\references{
    J. Joseph and R. Perisamy. ``An image driven bilateral filter with adaptive range and spatial parameters for denoising Magnetic Resonance Images.'' Computers and Electrical Engineering, vol. 69, pp. 782-795, July 2018.

    P. Perona and J. Malik. ``Scale-space and edge detection using anisotropic diffusion.'' IEEE Transactions on Pattern Analysis and Machine Intelligence, vol. 12, issue 7, pp. 629-639, July 1990.

    K. He, J. Sun, and X. Tang. ``Guided Image Filtering.'' IEEE Transactions on Pattern Analysis and Machine Intelligence, vol. 35, no. 6, pp. 1397-1409, June 2013.

    D. Liu and C. He. ``Peak-aware guided filtering for spectrum signal denoising.'' Chemometrics and Intelligent Laboratory Systems, vol. 222, March 2022.

    A. Savitzky and M. J. E. Golay. ``Smoothing and Differentiation of Data by Simplified Least Squares Procedures.'' Analytical Chemistry, vol. 36, no. 8, pp. 1627-1639, July 1964.
}

\examples{
set.seed(1)
t <- seq(from=0, to=6 * pi, length.out=5000)
y <- sin(t) + 0.6 * sin(2.6 * t)
x <- y + runif(length(y))
xs <- filt1_gauss(x, width=25)

plot(x, type="l")
lines(xs, col="red")
}

\keyword{ts}
\keyword{smooth}
