% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plotInteractions.R
\name{plotInteractions}
\alias{plotInteractions}
\title{Plot interaction graph}
\usage{
plotInteractions(
  out,
  object,
  label,
  group_by,
  node_color_by = NULL,
  node_size_by = NULL,
  node_color_fix = NULL,
  node_size_fix = NULL,
  node_label_repel = TRUE,
  node_label_color_by = NULL,
  node_label_color_fix = NULL,
  edge_color_by = NULL,
  edge_color_fix = NULL,
  edge_width_by = NULL,
  edge_width_fix = NULL,
  draw_edges = TRUE,
  return_data = FALSE,
  graph_layout = "circle"
)
}
\arguments{
\item{out}{a data frame, usually the output from \code{countInteractions} or 
\code{testInteractions}, representing an edge list with columns \code{"group_by",
"from_label" and "to_label"}. Additional columns may be included to specify 
edge attributes (weight or color).}

\item{object}{a \code{SingleCellExperiment} or \code{SpatialExperiment}
object.}

\item{label}{single character specifying the \code{colData(object)} entry
which stores the cell labels. These can be cell-types labels or other
metadata entries.}

\item{group_by}{a single character indicating the \code{colData(object)}
entry by which interactions are grouped. This is usually the image or patient ID. 
a single character indicating the \code{colData(object)}}

\item{node_color_by}{single character either
\code{NULL, "name","n_cells", "n_group"} by which the nodes should be
colored.}

\item{node_size_by}{single character either \code{NULL, "n_cells","n_group"} 
by which the size of the nodes are defined.}

\item{node_color_fix}{single character specifying the color of all nodes.}

\item{node_size_fix}{single numeric specifying the size of all nodes.}

\item{node_label_repel}{should nodes be labelled? Defaults to TRUE.}

\item{node_label_color_by}{single character either
\code{NULL, "name","n_cells","n_group"} by which the node labels should be
colored.}

\item{node_label_color_fix}{single character specifying the color of all node
labels.}

\item{edge_color_by}{single character indicating the name of the column of \code{"out"} 
used represent edge colors. This column is usually newly added by the user and must assign a 
unique value to each 'from_label'–'to_label' pair. Typically, these values could encode 
the direction of significantly interacting cell type pairs.}

\item{edge_color_fix}{single character specifying the color of all edges.}

\item{edge_width_by}{single character indicating the name of the column of \code{"out"} 
used to scale edge widths. The values in this column are averaged for each 
'from_label'–'to_label' pair. Typically, this could be the 'ct' column from 
of \code{"out"} or a newly added column representing an interaction feature.}

\item{edge_width_fix}{single numeric specifying the width of all edges.}

\item{draw_edges}{should edges be drawn between nodes? Defaults to TRUE.}

\item{return_data}{should the edge list and vertex metadata for graph
construction be returned as a \code{list} of two \code{data.frames}?}

\item{graph_layout}{single character of
\code{"circle", "chord", "linear", "fr", "kk", "drl", "stress", "graphopt", 
"lgl", "tree", "sugiyama", "star", "nicely", "manual", "grid", "mds", "sphere", 
"randomly", "gem", "dh"} which defines the graph layout.
Defaults to \code{"circle"}. For more information, see \link[ggraph]{ggraph}.}
}
\value{
returns a \code{ggplot} object or a \code{list} of two
\code{data.frames}.
}
\description{
Function to plot directed interaction graphs based on symbolic 
edge-lists and vertex metadata.
The user can specify node, node_label and edge aesthetics using dedicated
arguments. The resulting plot can be further refined with `ggplot2` for node 
styling and `ggraph` for edge-specific customization.
}
\examples{
set.seed(22)
library(cytomapper)
library(BiocParallel)
data(pancreasSCE)

## 1. countInteractions or testInteractions
sce  <- buildSpatialGraph(pancreasSCE, img_id = "ImageNb", type = "knn", k = 3)

count_out <- countInteractions(sce,
                               group_by = "ImageNb",
                               label = "CellType",
                               method = "classic", # choose from c("classic", "histocat", "patch", "interaction")
                               colPairName = "knn_interaction_graph")

test_out <- testInteractions(sce, 
                             group_by = "ImageNb",
                             label = "CellType", 
                             method = "classic", # choose from c("classic", "histocat", "patch", "interaction")
                             colPairName = "knn_interaction_graph", 
                             iter = 100, 
                             p_threshold = 0.5, 
                             BPPARAM = SerialParam(RNGseed = 123))

## 2. Plot interactions 

# default                
plotInteractions(count_out, sce, "CellType", "ImageNb")

# adjust node aesthetics
plotInteractions(count_out, sce, "CellType", "ImageNb",
                 node_color_by = "name",
                 node_size_by = "n_cells")
                 
# adjust edge aesthetics
plotInteractions(test_out, sce, "CellType", "ImageNb", 
                 edge_width_by = "ct")
                   
# Plot interactions - return data
plotInteractions(test_out, sce, "CellType", "ImageNb",
                 return_data = TRUE)          
                  
}
\seealso{
\code{\link{countInteractions}} for counting (but not testing) cell-cell
interactions per grouping level.
\code{\link{testInteractions}} for testing cell-cell 
interactions per grouping level.
}
\author{
Marlene Lutz (\email{marlene.lutz@uzh.ch})
}
