% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/iSEE-main.R
\name{iSEE}
\alias{iSEE}
\title{iSEE: interactive SummarizedExperiment Explorer}
\usage{
iSEE(
  se,
  initial = NULL,
  extra = NULL,
  colormap = ExperimentColorMap(),
  landingPage = createLandingPage(),
  tour = NULL,
  appTitle = NULL,
  tabTitle = NULL,
  runLocal = TRUE,
  voice = FALSE,
  bugs = FALSE,
  saveState = NULL,
  ...
)
}
\arguments{
\item{se}{A \linkS4class{SummarizedExperiment} object, ideally with named assays.
If missing, an app is launched with a landing page generated by the \code{landingPage} argument.}

\item{initial}{A list of \linkS4class{Panel} objects specifying the initial state of the app.
The order of panels determines the sequence in which they are laid out in the interface.
Defaults to one instance of each panel class available from \pkg{iSEE}.}

\item{extra}{A list of additional \linkS4class{Panel} objects that might be added after the app has started.
Defaults to one instance of each panel class available from \pkg{iSEE}.}

\item{colormap}{An \linkS4class{ExperimentColorMap} object that defines custom colormaps to apply to individual \code{assays}, \code{colData} and \code{rowData} covariates.}

\item{landingPage}{A function that renders a landing page when \code{iSEE} is started without any specified \code{se}.
Ignored if \code{se} is supplied.}

\item{tour}{A data.frame with the content of the interactive tour to be displayed after starting up the app.
Ignored if \code{se} is not supplied.}

\item{appTitle}{A string indicating the title to be displayed in the app.
If not provided, the app displays the version info of \code{\link{iSEE}}.}

\item{tabTitle}{A string indicating the title to be displayed in the browser
tab. If not provided, the tab is named \code{"iSEE"}.}

\item{runLocal}{A logical indicating whether the app is to be run locally or remotely on a server, which determines how documentation will be accessed.}

\item{voice}{A logical indicating whether the voice recognition should be enabled.}

\item{bugs}{Set to \code{TRUE} to enable the bugs Easter egg.
Alternatively, a named numeric vector control the respective number of each bug type (e.g., \code{c(bugs=3L, spiders=1L)}).}

\item{saveState}{A function that accepts a single argument containing the current application state and saves it to some appropriate location.}

\item{...}{Further arguments to pass to \code{\link{shinyApp}}.}
}
\value{
A Shiny app object is returned for interactive data exploration of \code{se},
either by simply printing the object or by explicitly running it with \code{\link{runApp}}.
}
\description{
Interactive and reproducible visualization of data contained in a
\linkS4class{SummarizedExperiment} object, using a Shiny interface.
}
\details{
Configuring the initial state of the app is as easy as passing a list of \linkS4class{Panel} objects to \code{initial}.
Each element represents one panel and is typically constructed with a command like \code{\link{ReducedDimensionPlot}()}.
Panels are filled from left to right in a row-wise manner depending on the available width.
Each panel can be easily customized by modifying the parameters in each object.

The \code{extra} argument should specify \linkS4class{Panel} classes that might not be shown during initialization
but can be added interactively by the user after the app has started.
The first instance of each new class in \code{extra} will be used as a template when the user adds a new panel of that class.
Note that \code{initial} will automatically be appended to \code{extra} to form the final set of available panels,
so it is not strictly necessary to re-specify instances of those initial panels in \code{extra}.
(unless we want the parameters of newly created panels to be different from those at initialization).
}
\section{Setting up a tour}{

The \code{tour} argument allows users to specify a custom tour to walk their audience through various panels.
This is useful for describing different aspects of the dataset and highlighting interesting points in an interactive manner.

We use the format expected by the \code{rintrojs} package - see \url{https://github.com/carlganz/rintrojs#usage} for more information.
There should be two columns, \code{element} and \code{intro}, with the former describing the element to highlight and the latter providing some descriptive text. 
The \code{\link{defaultTour}} also provides the default tour that is used in the Examples below.
}

\section{Creating a landing page}{

If \code{se} is not supplied, a landing page is generated that allows users to upload their own RDS file to initialize the app.
By default, the maximum request size for file uploads defaults to 5MB
(\url{https://shiny.rstudio.com/reference/shiny/0.14/shiny-options.html}).
To raise the limit (e.g., 50MB), run \code{options(shiny.maxRequestSize=50*1024^2)}.

The \code{landingPage} argument can be used to alter the landing page, see \code{\link{createLandingPage}} for more details.
This is useful for creating front-ends that can retrieve \linkS4class{SummarizedExperiment}s from a database on demand for interactive visualization.
}

\section{Saving application state}{

If users want to record the application state, they can download an RDS file containing a list with the entries:
\itemize{
\item \code{memory}, a list of \linkS4class{Panel} objects containing the current state of the application.
This can be directly re-used as the \code{initial} argument in a subsequent \code{\link{iSEE}} call.
\item \code{se}, the \linkS4class{SummarizedExperiment} object of interest.
This is optional and may not be present in the list, depending on the user specifications.
\item \code{colormap}, the \linkS4class{ExperimentColorMap} object being used.
This is optional and may not be present in the list, depending on the user specifications.
}

We can also provide a custom function in \code{saveState} that accepts a single argument containing this list.
This is most useful when \code{\link{iSEE}} is deployed in an enterprise environment where sessions can be saved in a persistent location;
combined with a suitable \code{landingPage} specification, this allows users to easily reload sessions of interest.
The idea is very similar to Shiny bookmarks but is more customizable and can be used in conjunction with URL-based bookmarking.
}

\examples{
library(scRNAseq)

# Example data ----
sce <- ReprocessedAllenData(assays="tophat_counts")
class(sce)

library(scater)
sce <- logNormCounts(sce, exprs_values="tophat_counts")

sce <- runPCA(sce, ncomponents=4)
sce <- runTSNE(sce)
rowData(sce)$ave_count <- rowMeans(assay(sce, "tophat_counts"))
rowData(sce)$n_cells <- rowSums(assay(sce, "tophat_counts") > 0)
sce

# launch the app itself ----

app <- iSEE(sce)
if (interactive()) {
  shiny::runApp(app, port=1234)
}

}
\references{
Rue-Albrecht K, Marini F, Soneson C, Lun ATL.
iSEE: Interactive SummarizedExperiment Explorer
\emph{F1000Research} 7.

Javascript code for \code{bugs} was based on \url{https://github.com/Auz/Bug}.
}
