% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/IO.R
\name{read.FCS}
\alias{read.FCS}
\alias{cleanup}
\alias{isFCSfile}
\title{Read an FCS file}
\usage{
isFCSfile(files)

read.FCS(filename, transformation="linearize", which.lines=NULL,
         alter.names=FALSE, column.pattern=NULL, invert.pattern = FALSE,
         decades=0, ncdf = FALSE, min.limit=NULL, 
         truncate_max_range = TRUE, dataset=NULL, emptyValue=TRUE, 
         channel_alias = NULL, ...)
}
\arguments{
\item{filename}{Character of length 1: filename}

\item{transformation}{An character string that defines the type of
transformation. Valid values are \code{linearize} (default),
\code{linearize-with-PnG-scaling}, or \code{scale}.  The \code{linearize}
transformation applies the appropriate power transform to the data. The
\code{linearize-with-PnG-scaling} transformation applies the appropriate
power transform for parameters stored on log scale, and also a linear
scaling transformation based on the 'gain' (FCS \$PnG keywords) for
parameters stored on a linear scale. The \code{scale} transformation scales
all columns to $[0,10^decades]$.  defaulting to decades=0 as in the FCS4
specification.  A logical can also be used: \code{TRUE} is equal to
\code{linearize} and \code{FALSE}(or \code{NULL}) corresponds to no
transformation.  Also when the transformation keyword of the FCS header is
set to "custom" or "applied", no transformation will be used.}

\item{which.lines}{Numeric vector to specify the indices of the lines to be
read. If NULL all the records are read, if of length 1, a random sample of
the size indicated by \code{which.lines} is read in. It's used to achieve partial disk IO
for the large FCS that can't fit the full data into memory. Be aware the potential slow read
(especially for the large size of random sampling) due to the frequent disk seek operations.}

\item{alter.names}{boolean indicating whether or not we should rename the
columns to valid R names using \code{\link{make.names}}. The default is
FALSE.}

\item{column.pattern}{An optional regular expression defining parameters we
should keep when loading the file. The default is NULL.}

\item{invert.pattern}{logical. By default, \code{FALSE}. If \code{TRUE},
inverts the regular expression specified in \code{column.pattern}. This is
useful for indicating the channel names that we do not want to read. If
\code{column.pattern} is set to \code{NULL}, this argument is ignored.}

\item{decades}{When scaling is activated, the number of decades to use for
the output.}

\item{ncdf}{Deprecated. Please use 'ncdfFlow' package for cdf based storage.}

\item{min.limit}{The minimum value in the data range that is allowed. Some
instruments produce extreme artifactual values. The positive data range for
each parameter is completely defined by the measurement range of the
instrument and all larger values are set to this threshold. The lower data
boundary is not that well defined, since compensation might shift some
values below the original measurement range of the instrument. This can be 
set to an arbitrary number or to \code{NULL} (the default value), in which 
case the original values are kept. When the transformation keyword of the FCS header is
set (typically to "custom" or "applied"), no shift up to min.limit will occur.}

\item{truncate_max_range}{logical type. Default is TRUE. can be optionally
turned off to avoid truncating the extreme positive value to the instrument
measurement range .i.e.'$PnR'. When the transformation keyword of the FCS header is
set (typically to "custom" or "applied"), no truncation will occur.}

\item{dataset}{The FCS file specification allows for multiple data segments
in a single file. Since the output of \code{read.FCS} is a single
\code{flowFrame} we can't automatically read in all available sets. This
parameter allows to chose one of the subsets for import. Its value is
supposed to be an integer in the range of available data sets. This argument
is ignored if there is only a single data segment in the FCS file.}

\item{emptyValue}{boolean indicating whether or not we allow empty value for
keyword values in TEXT segment.  It affects how the double delimiters are
treated.  IF TRUE, The double delimiters are parsed as a pair of start and
end single delimiter for an empty value.  Otherwise, double delimiters are
parsed one part of string as the keyword value.  default is TRUE.}

\item{channel_alias}{an optional data.frame used to provide the alias of the channels
to standardize and solve the discrepancy across FCS files. It is expected to
contain 'alias' and 'channels' column of 'channel_alias'. Each row/entry specifies the common
alias name for a collection of channels (comma separated). See examples for
details. 

For each channel in the FCS file, read.FCS will first attempt
to find an exact match in the 'channels' column. If no exact match is found,
it will check for partial matches. That is, if "V545" is in the 'channels'
column of 'channel_alias' and "V545-A" is present in the FCS file, this
partial match will allow the corresponding 'alias' to be assigned. This partial
matching only works in this direction ("V545-A" in the 'channels' column will
not match "V545" in the FCS file) and care should be exercised to ensure no unintended
partial matching of other channel names. If no exact or partial match is found, 
the channel is unchanged in the resulting \code{flowFrame}.}

\item{...}{ignore.text.offset: whether to ignore the keyword values in TEXT
segment when they don't agree with the HEADER.  Default is FALSE, which
throws the error when such discrepancy is found.  User can turn it on to
ignore TEXT segment when he is sure of the accuracy of HEADER so that the
file still can be read.}

\item{files}{A vector of filenames}
}
\value{
\code{isFCSfile} returns a logical vector.

\code{read.FCS} returns an object of class
\code{\link[flowCore:flowFrame-class]{flowFrame}} that contains the data in
the \code{exprs} slot, the parameters monitored in the \code{parameters}
slot and the keywords and value saved in the header of the FCS file.
}
\description{
Check validity and Read Data File Standard for Flow Cytometry
}
\details{
The function \code{isFCSfile} determines whether its arguments are valid FCS
files.

The function \code{read.FCS} works with the output of the FACS machine
software from a number of vendors (FCS 2.0, FCS 3.0 and List Mode Data LMD).
However, the FCS 3.0 standard includes some options that are not yet
implemented in this function. If you need extensions, please let me know.
The output of the function is an object of class \code{flowFrame}.

For specifications of FCS 3.0 see \url{http://www.isac-net.org} and the file
\url{../doc/fcs3.html} in the \code{doc} directory of the package.

The \code{which.lines} arguments allow you to read a subset of the record as
you might not want to read the thousands of events recorded in the FCS file.
It is mainly used when there is not enough memory to read one single FCS
(which probably will not happen).  It will probably take more time than
reading the entire FCS (due to the multiple disk IO).
}
\examples{

## a sample file
fcsFile <- system.file("extdata", "0877408774.B08", package="flowCore")

## read file and linearize values
samp <-  read.FCS(fcsFile, transformation="linearize")
exprs(samp[1:3,])
keyword(samp)[3:6]
class(samp)

## Only read in lines 2 to 5
subset <- read.FCS(fcsFile, which.lines=2:5, transformation="linearize")
exprs(subset)

## Read in a random sample of 100 lines
subset <- read.FCS(fcsFile, which.lines=100, transformation="linearize")
nrow(subset)

#manually supply the alias vs channel options mapping as a data.frame
map <- data.frame(alias = c("A", "B")
                  , channels = c("FL2", "FL4")
)
fr <- read.FCS(fcsFile, channel_alias = map)
fr

}
\seealso{
\code{\link[flowCore]{read.flowSet}}
}
\author{
F. Hahne, N.Le Meur
}
\keyword{IO}
