% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/runEscape.R
\name{escape.matrix}
\alias{escape.matrix}
\title{Calculate Single-Cell Gene-Set Enrichment Scores}
\usage{
escape.matrix(
  input.data,
  gene.sets = NULL,
  method = "ssGSEA",
  groups = 1000,
  min.size = 5,
  normalize = FALSE,
  make.positive = FALSE,
  min.expr.cells = 0,
  min.filter.by = NULL,
  BPPARAM = NULL,
  ...
)
}
\arguments{
\item{input.data}{A raw-counts matrix (genes x cells), a
\link[SeuratObject]{Seurat} object, or a
\link[SingleCellExperiment]{SingleCellExperiment}. Gene identifiers must
match those in \code{gene.sets}.}

\item{gene.sets}{A named list of character vectors, the result of
\code{\link{getGeneSets}}, or the built-in data object
\code{\link{escape.gene.sets}}. List names become column names in the
result.}

\item{method}{Character. Scoring algorithm (case-insensitive). One of
\code{"GSVA"}, \code{"ssGSEA"}, \code{"UCell"}, or \code{"AUCell"}.
Default is \code{"ssGSEA"}.}

\item{groups}{Integer. Number of cells per processing chunk. Larger values
reduce overhead but increase memory usage. Default is \code{1000}.}

\item{min.size}{Integer or \code{NULL}. Minimum number of genes from a set
that must be detected in the expression matrix for that set to be scored.
Default is \code{5}. Use \code{NULL} to disable filtering.}

\item{normalize}{Logical. If \code{TRUE}, the score matrix is passed to
\code{\link{performNormalization}} (drop-out scaling and optional log
transform). Default is \code{FALSE}.}

\item{make.positive}{Logical. If \code{TRUE} \emph{and}
\code{normalize = TRUE}, shifts every gene-set column so its global
minimum is zero, facilitating downstream log-ratio analyses. Default is
\code{FALSE}.}

\item{min.expr.cells}{Numeric. Gene-expression filter threshold. Default is
\code{0} (no gene filtering).}

\item{min.filter.by}{Character or \code{NULL}. Column name in
\code{meta.data} (Seurat) or \code{colData} (SCE) defining groups within
which the \code{min.expr.cells} rule is applied. Default is \code{NULL}.}

\item{BPPARAM}{A \pkg{BiocParallel} parameter object describing the
parallel backend. Default is \code{NULL} (serial execution).}

\item{...}{Extra arguments passed verbatim to the chosen back-end scoring
function (\code{gsva()}, \code{ScoreSignatures_UCell()}, or
\code{AUCell_calcAUC()}).}
}
\value{
A numeric matrix with one row per cell and one column per gene set,
  ordered as in \code{gene.sets}.
}
\description{
\code{escape.matrix()} computes per-cell enrichment for arbitrary gene-set
collections using one of four scoring back-ends and returns a dense numeric
matrix (cells x gene-sets). The expression matrix is processed in
user-defined chunks (\code{groups}) so that memory use remains predictable;
each chunk is dispatched in parallel via a \pkg{BiocParallel} \code{BPPARAM}
backend. Heavy engines (\pkg{GSVA}, \pkg{UCell}, \pkg{AUCell}) are loaded
lazily, keeping them in the package's \strong{Suggests} field.
}
\section{Supported methods}{

\describe{
  \item{\code{"GSVA"}}{Gene-set variation analysis (Poisson kernel).}
  \item{\code{"ssGSEA"}}{Single-sample GSEA.}
  \item{\code{"UCell"}}{Rank-based UCell scoring.}
  \item{\code{"AUCell"}}{Area-under-the-curve ranking score.}
}
}

\examples{
gs <- list(Bcells = c("MS4A1", "CD79B", "CD79A", "IGH1", "IGH2"),
           Tcells = c("CD3E", "CD3D", "CD3G", "CD7","CD8A"))

pbmc <- SeuratObject::pbmc_small
es   <- escape.matrix(pbmc, 
                      gene.sets = gs,
                      method = "ssGSEA", 
                      groups = 500, 
                      min.size = 3)

}
\seealso{
\code{\link{runEscape}} to attach scores to a single-cell object;
  \code{\link{getGeneSets}} for MSigDB retrieval;
  \code{\link{performNormalization}} for the optional normalization workflow.
}
\author{
Nick Borcherding, Jared Andrews
}
