% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calculateActivity.R
\name{calculateActivity}
\alias{calculateActivity}
\title{Calculate the per cell activity of master regulators based on a regulon}
\usage{
calculateActivity(
  expMatrix = NULL,
  exp_assay = "logcounts",
  regulon = NULL,
  normalize = FALSE,
  mode = "weight",
  method = deprecated(),
  genesets = NULL,
  clusters = NULL,
  FUN = c("mean", "sum")
)
}
\arguments{
\item{expMatrix}{A SingleCellExperiment object containing gene expression information with rows representing genes and columns represent cells.
Rownames (either gene symbols or geneID) must be consistent with the naming convention in the regulon.}

\item{exp_assay}{String specifying the name of the assay to be retrieved from the SingleCellExperiment object. Set to
'logcounts' as the default}

\item{regulon}{A DataFrame object consisting of tf (regulator) and target in the column names, with additional columns
indicating degree of association between tf and target such as 'mor' or 'corr' obtained from \code{addWeights}.}

\item{normalize}{Logical indicating whether row means should be subtracted from expression matrix. default is FALSE}

\item{mode}{String indicating the name of column to be used as the weights}

\item{genesets}{A list of genesets. Each list element can be a dataframe with the first column indicating the genes and second column indicating the weights.
Alternatively, each list element is a character vector corresponding to the genes in the geneset. A feature set collection in the form of CompressedSplitDataFrameList that
contains genes in the first column and weights in the second column. See details}

\item{clusters}{A vector indicating cluster assignment}

\item{FUN}{function to aggregate the weights}
}
\value{
A matrix of inferred transcription factor (row) activities in single cells (columns)
}
\description{
Calculate the per cell activity of master regulators based on a regulon
}
\details{
This function calculates activity score from a regulon that is a DataFrame consisting of a tf column,
a target column and a weight column. Alternatively, instead of a regulon, this function also accepts weighted
signature sets where each gene set or signature is a data frame or unweighted signature sets where each gene set is a character vector.
The user has the option of computing signature score by weighted mean of target gene expression or
the relative ranking of the target genes computed by AUCell.
}
\examples{
# create a mock singleCellExperiment object for gene expMatrixession matrix
set.seed(1000)
gene_sce <- scuttle::mockSCE()
gene_sce <- scuttle::logNormCounts(gene_sce)
rownames(gene_sce) <- paste0('Gene_',1:2000)

# create a mock singleCellExperiment object for peak matrix
peak_gr <- GRanges(seqnames = 'chr1',
                   ranges = IRanges(start = seq(from = 1, to = 10000, by = 100), width = 100))
peak_counts <- matrix(sample(x = 0:4, size = ncol(gene_sce)*length(peak_gr), replace = TRUE),
                      nrow = length(peak_gr), ncol=ncol(gene_sce))
peak_sce <- SingleCellExperiment(list(counts = peak_counts), colData = colData(gene_sce))
rownames(peak_sce) <- paste0('Peak_',1:100)

# create a mock regulon
regulon <- data.frame(tf = c(rep('Gene_1',10), rep('Gene_2',10)),
                      idxATAC = sample(1:100, 20),
                      target = c(paste0('Gene_', sample(3:2000,10)),
                                 paste0('Gene_',sample(3:2000,10))))

#  # prune regulon
pruned.regulon <- pruneRegulon(expMatrix = gene_sce,
                               exp_assay = 'logcounts',
                               peakMatrix = peak_sce,
                               peak_assay = 'counts',
                               regulon = regulon,
                               clusters = gene_sce$Treatment,
                               regulon_cutoff = 0.5,
                               p_adj = TRUE)

regulon.w <- addWeights(regulon = regulon,
                        expMatrix = gene_sce,
                        clusters = gene_sce$Treatment,
                        exp_assay = 'logcounts',
                        min_targets = 5,
                        method = 'corr')

# calculate activity
activity <- calculateActivity(expMatrix = gene_sce,
                              regulon = regulon.w,
                              exp_assay = 'logcounts')

# calculate cluster-specific activity if cluster-specific weights are supplied
regulon.w$weight <- matrix(runif(nrow(regulon.w)*2, -1,1), nrow(regulon.w),2)
colnames(regulon.w$weight) <- c('treat1','treat2')

activity.cluster <- calculateActivity(gene_sce,
regulon = regulon.w, clusters = gene_sce$Treatment,
exp_assay = 'logcounts', FUN = 'mean')

# compute signature scores from weighted genesets
weighted_genesets <- list(set1 = data.frame(genes = c('Gene_1', 'Gene_2', 'Gene_3'),
weights = c(1,2,3)), set2 = data.frame(genes = c('Gene_4', 'Gene_5', 'Gene_6'), weights = c(4,5,6)))

activity <- calculateActivity(gene_sce, genesets = weighted_genesets)

# compute signature scores from unweighted genesets
unweighted_genesets <- list(set1 = c('Gene_1', 'Gene_2', 'Gene_3'),
                            set2 = c('Gene_4', 'Gene_5', 'Gene_6'))
activity <- calculateActivity(gene_sce, genesets = unweighted_genesets)

}
\author{
Xiaosai Yao, Shang-yang Chen
}
