% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/graphs.R
\name{buildGraph}
\alias{buildGraph}
\alias{buildDiffGraph}
\alias{addCentrality}
\alias{normalizeCentrality}
\alias{rankTfs}
\title{Creating graphs and related operations}
\usage{
buildGraph(
  regulon,
  mode = c("tg", "tripartite", "re", "pairs"),
  weights = "weights",
  cluster = "all",
  aggregation_function = function(x) x[which.max(abs(x))],
  na_replace = TRUE,
  keep_original_names = TRUE,
  filter_edges = NULL
)

buildDiffGraph(graph_obj_1, graph_obj_2, weighted = TRUE, abs_diff = TRUE)

addCentrality(graph)

normalizeCentrality(graph, FUN = sqrt, weighted = TRUE)

rankTfs(graph, type_attr = "type")
}
\arguments{
\item{regulon}{an object returned by the getRegulon or addWeights function.}

\item{mode}{a character specifying which type of graph will be built. In \code{'tg'} mode
a bipartite graph is built by connecting transcription factors directly to the target genes
and ignoring information about mediating regulatory elements; in \code{'pairs'} mode
transcription factors are connected to unique target gene-regulatory element pairs;
in \code{'tripartite'} mode the network is made up of three types of vertices (nodes):
transcription factors, regulatory elements and target genes; here the path from
target gene to regulatory element always contains a regulatory element; in
\code{'re'} mode data in the target genes is dropped and only connections are
between transcription factors and regulatory elements.}

\item{weights}{a character specifying which variable should be used to assign
weights to edges.}

\item{cluster}{a character specifying the name of the cluster column which should be used
to retrieve weight values from regulon object. Using this argument makes sense
only with combination with \code{weights} parameter when it points to the regulon column
that is a matrix.}

\item{aggregation_function}{a function used to aggregate weights of duplicated edges,
which might appear due to the many transcription factor converging at the same regulatory
element; starting from this point each transcription factor is supposed to have a separate
connection to the target gene, perhaps the same one across several connections. In
\code{tripartite} mode this might result in many edges in the same node pair, however weights might
differ since they are inherited from different tf-re-tg triplets (rows) in the
\code{regulon} object. Similarly, duplicated edges are generated by one
transcription factor using a regulatory element multiple times to reach different
target genes. In \code{tg} mode the edges became duplicated if one transcription
factor reaches the same target genes through many regulatory elements.}

\item{na_replace}{a logical indicating whether NA values for weights should be
replaced with zeros.}

\item{keep_original_names}{A logical indicating whether gene names should be used as
node names in the output graph. Note that this might lead to the duplicated node
names if the same gene is present in two layers (transcription factors and target genes).}

\item{filter_edges}{A numeric defining the cutoff weight used for filtering out edges
which have weights equal or greater than cutoff. The isolated vertices are removed then
from the graph. Defaults to NULL in which case no filtering is applied.}

\item{weighted}{a logical indicating whether weighted graphs are used; in \code{tripartite}
mode tf-re-tg triplet is decomposed into two edges corresponding to tf-re and re-tg pairs,
and both edges inherit the same weight, which was originally assigned to the parent triplet.}

\item{abs_diff}{a logical indicating whether absolute difference in the number
edges or their weights will be calculated.}

\item{graph, graph_obj_1, graph_obj_2}{an igraph object.}

\item{FUN}{a function used for normalization. The input to this
function is be the number of edges connected with each node (incident edges).}

\item{type_attr}{a character corresponding to the name of the vertex attribute
which indicate the type of vertex.}
}
\value{
an igraph object.
\code{rankTfs} returns a data.frame with transcription factors sorted according
to the value of the \code{centrality} attribute.
}
\description{
The function enable to create graph objects using as input regulon objects returned by
\code{pruneRegulon} or \code{addWeights}. Both weighted and unweighted graphs can be
created that can further be visualized using dedicated functions.
}
\details{
\code{buildGraph} function creates a directed graph based on the output of
the \code{getRegulon} function. Four modes are available: (1) \code{tg} in which
connections are made directly between transcription factor and target genes. Even if
the same tf-tg pair is connected in the original regulon object through many
regulatory elements then only one edge is created. In such a case, when weighted
graph is created, weights are summarized by the aggregating function (by default
the maximum absolute value with the sign of the original value). Similarly, aggregation
is made in the \code{re} mode leaving only unique transcription factor-regulatory element pairs.
In \code{tripartite} mode edges connect transcription factors with regulatory elements and
regulatory elements with target genes. The same weights are used for both edges
that correspond to the single row in the regulon data frame (tf-re and re-tg). Note
that the original regulon structure is not fully preserved because each row is now
represented by two edges which are independent from each other. Thus they can be
coupled with different edges connected to the same regulatory element building the
path from transcription factor to the target gene of another transcription factor
through the shared regulatory element.

\code{buildDiffGraph} a graph difference by subtracting the edges of \code{graph_obj_2}
from those of the \code{graph_obj_1}. If \code{weighted} is set to \code{TRUE} then for each
ordered pair of vertices (nodes) the difference in number of edges between \code{graph_obj_1}
and \code{graph_obj_1} is calculated. The result is used to set the number of
corresponding edges in output graph. Note that unless \code{abs_diff} is set to
\code{TRUE} any non-positive difference will translate into lack of the edges
for a corresponding ordered pair of vertices in the output graph (equivalent
to 0 value in the respective position in adjacency matrix). In case of
weighted graphs, the weight of the output graph is calculated as a difference
of the corresponding weights between input graphs.

\code{addCentrality} calculates degree centrality for each vertex using
\code{igraph::strength}.

With \code{normalizeCentrality} function the normalized values of centrality
are calculated from the original ones divided by
\code{FUN}(total number of non-zero edges associated with each node).

\code{rankTfs} assign ranks to transcription factors according to degree
centrality of their vertices
}
\examples{
# create an artificial getRegulon output
set.seed(1234)
tf_set <- apply(expand.grid(LETTERS[1:10], LETTERS[1:10]),1,  paste, collapse = '')
regulon <- DataFrame(tf = sample(tf_set, 5e3, replace = TRUE))
gene_set <- expand.grid(LETTERS[1:10], LETTERS[1:10], LETTERS[1:10])
gene_set <- apply(gene_set,1,function(x) paste0(x,collapse=''))
regulon$target <- sample(gene_set, 5e3, replace = TRUE)
regulon$idxATAC <- 1:5e3
regulon$corr <- runif(5e3)*0.5+0.5
regulon$weights <- matrix(runif(15000), nrow=5000, ncol=3)
colnames(regulon$weights) <- c('all','cluster1', 'cluster2')
graph_tripartite <- buildGraph(regulon, cluster='all', mode = 'tripartite')

# build bipartite graph using regulatory element-target gene pairs
graph_pairs_1 <- buildGraph(regulon, cluster = 'cluster1', mode = 'pairs')
graph_pairs_2 <- buildGraph(regulon, cluster = 'cluster2', mode = 'pairs')
graph_diff <- buildDiffGraph(graph_pairs_1, graph_pairs_2)
graph_diff <- addCentrality(graph_diff)
graph_diff <- normalizeCentrality(graph_diff)
tf_ranking <- rankTfs(graph_diff)

}
