% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/getConcordants.R
\name{getConcordants}
\alias{getConcordants}
\title{Get concordant signatures from iLINCS database
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#stable}{\figure{lifecycle-stable.svg}{options: alt='[Stable]'}}}{\strong{[Stable]}}}
\usage{
getConcordants(signature, ilincsLibrary = "CP")
}
\arguments{
\item{signature}{A data.frame, tibble, or S4Vectors::DataFrame containing
the signature data. Must conform to iLINCS signature structure with columns:
* \code{signatureID}: Signature identifier
* \code{ID_geneid}: Gene IDs
* \code{Name_GeneSymbol}: Gene symbols
* \code{Value_LogDiffExp}: Log fold-change values
* \code{Significance_pvalue}: Statistical significance p-values

Use \code{\link[=prepareSignature]{prepareSignature()}} to ensure proper formatting.}

\item{ilincsLibrary}{Character string specifying the iLINCS library to search.
Must be one of:
* \code{"CP"}: Chemical Perturbagen library (default)
* \code{"KD"}: Knockdown library
* \code{"OE"}: Overexpression library}
}
\value{
A data structure containing concordant signatures. The return type
matches the input signature type:
* tibble for data.frame or tibble inputs
* S4Vectors::DataFrame for DataFrame inputs

Contains the following columns:
* \code{signatureid}: Unique signature identifier
* \code{compound} or \code{treatment}: Drug/treatment name
* \code{concentration}: Drug concentration (CP library only)
* \code{time}: Treatment duration
* \code{cellline}: Cell line used
* \code{similarity}: Similarity score (rounded to 8 decimal places)
* \code{pValue}: Statistical significance (rounded to 20 decimal places)
* \code{sig_direction}: Signature direction ("Up", "Down", or "Any")
}
\description{
This function queries the iLINCS (Integrative Library of Integrated
Network-based Cellular Signatures) database to find signatures that are
concordant (similar) to a given input signature.
}
\details{
The function performs the following steps:
\enumerate{
\item Validates input parameters
\item Creates a temporary file with signature data
\item Detects signature direction from expression values
\item Sends a multipart POST request to the iLINCS API
\item Processes the JSON response into a standardized tibble
\item Cleans up temporary files
}

The signature direction is determined as follows:
\itemize{
\item \code{"Up"}: All expression values are greater than or equal to zero
\item \code{"Down"}: All expression values are less than or equal to zero
\item \code{"Any"}: Mixed positive and negative values
}
}
\section{API Details}{

This function interfaces with the iLINCS web service API. The signature
is uploaded as a tab-separated file and analyzed against the specified
library. Results are returned as JSON and parsed into a tibble.
}

\section{Error Handling}{

The function will stop execution with informative error messages for:
\itemize{
\item Invalid signature data types (must be data.frame, tibble, or DataFrame)
\item Invalid signature structure (missing required columns, wrong order, etc.)
\item Missing values in signature data
\item Unsupported iLINCS library names
\item HTTP errors from the iLINCS API
\item Invalid or empty API responses
}
}

\examples{
# Input validation examples (no API calls)
# These demonstrate proper signature structure
mockSig <- data.frame(
    signatureID = rep("TEST", 3),
    ID_geneid = c("123", "456", "789"),
    Name_GeneSymbol = c("TP53", "MYC", "EGFR"),
    Value_LogDiffExp = c(1.5, -2.0, 0.8)
)

# Validate library parameter (should produce error)
tryCatch(
    getConcordants(mockSig, ilincsLibrary = "INVALID"),
    error = function(e) message("Expected error: invalid library")
)

# This example requires network access to the iLINCS API

# Load example differential expression data
dge_file <- system.file("extdata", "dCovid_diffexp.tsv",
    package = "drugfindR"
)
dge_data <- read.delim(dge_file)

# Prepare signature to ensure proper structure
signature <- prepareSignature(
    dge_data[1:50, ],
    geneColumn = "hgnc_symbol",
    logfcColumn = "logFC",
    pvalColumn = "PValue"
)

# Find concordant chemical perturbagens
cpConcordants <- getConcordants(signature, ilincsLibrary = "CP")
head(cpConcordants)

# Find concordant knockdown signatures
kdConcordants <- getConcordants(signature, ilincsLibrary = "KD")
head(kdConcordants)

# Find concordant overexpression signatures
oeConcordants <- getConcordants(signature, ilincsLibrary = "OE")
head(oeConcordants)

# Works with different data frame types
signatureDf <- as.data.frame(signature)
cpConcordantsDf <- getConcordants(signatureDf, "CP")

# Works with S4Vectors::DataFrame
signatureDataFrame <- S4Vectors::DataFrame(signature)
cpConcordantsDataFrame <- getConcordants(signatureDataFrame, "CP")
# Returns S4Vectors::DataFrame to match input type

}
\references{
iLINCS Portal: \url{http://www.ilincs.org/}

Pilarczyk et al. (2020). Connecting omics signatures and revealing
biological mechanisms with iLINCS. Nature Communications, 11(1), 4058.
}
\seealso{
\verb{[ prepareSignature() ]} for signature preparation,
\verb{[ filterSignature() ]} for signature filtering,
\verb{[ investigateSignature() ]} for signature investigation
}
