% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/filterSignature.R
\name{.applyDirectionFilter}
\alias{.applyDirectionFilter}
\title{Apply filtering based on direction and thresholds}
\usage{
.applyDirectionFilter(signature, thresholds, direction = "any")
}
\arguments{
\item{signature}{A data.frame-like object containing the signature data.
Must have a column named "Value_LogDiffExp" containing log fold-change values.}

\item{thresholds}{A named list containing:
\itemize{
\item \code{downThreshold}: Threshold for down-regulated genes
\item \code{upThreshold}: Threshold for up-regulated genes
}}

\item{direction}{Character string specifying the filtering direction.
Must be one of:
* "up": Keep only up-regulated genes (logFC >= upThreshold)
* "down": Keep only down-regulated genes (logFC <= downThreshold)
* "any": Keep both up- and down-regulated genes (logFC >= upThreshold OR logFC <= downThreshold)}
}
\value{
A tibble containing the filtered signature data with the same structure
as the input but including only rows that meet the filtering criteria.
}
\description{
This internal function performs the actual filtering of the signature data
based on the specified direction and calculated thresholds. It implements
the core filtering logic using dplyr operations.
}
\details{
The filtering logic depends on the direction parameter:
\itemize{
\item \code{"up"}: Retains genes where \code{Value_LogDiffExp >= upThreshold}
\item \code{"down"}: Retains genes where \code{Value_LogDiffExp <= downThreshold}
\item \code{"any"}: Retains genes where
\verb{Value_LogDiffExp >= upThreshold OR Value_LogDiffExp <= downThreshold}
}

The function uses \code{dplyr::filter} with \code{rlang::.data} for
non-standard evaluation, ensuring compatibility with different data frame types
and avoiding issues with variable scoping.
}
\examples{
\dontrun{
# Create sample signature data
signature <- data.frame(
    signatureID = rep("TEST", 10),
    Name_GeneSymbol = paste0("GENE", 1:10),
    Value_LogDiffExp = c(-3, -2, -1, -0.5, 0, 0.5, 1, 2, 3, 4)
)

# Define thresholds
thresholds <- list(downThreshold = -1.5, upThreshold = 1.5)

# Filter for up-regulated genes only
up_filtered <- .applyDirectionFilter(signature, "up", thresholds)
# Returns genes with logFC >= 1.5 (GENE8, GENE9, GENE10)

# Filter for down-regulated genes only
down_filtered <- .applyDirectionFilter(signature, "down", thresholds)
# Returns genes with logFC <= -1.5 (GENE1, GENE2)

# Filter for both up- and down-regulated genes
both_filtered <- .applyDirectionFilter(signature, "any", thresholds)
# Returns genes with |logFC| >= 1.5 (GENE1, GENE2, GENE8, GENE9, GENE10)
}
}
\keyword{internal}
