% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/getDeepHFScores.R
\name{getDeepHFScores}
\alias{getDeepHFScores}
\title{Calculate on-target sgRNA activity scores for Cas9 using DeepHF}
\usage{
getDeepHFScores(
  sequences,
  enzyme = c("WT", "ESP", "HF"),
  promoter = c("U6", "T7"),
  fork = FALSE
)
}
\arguments{
\item{sequences}{Character vector of 23bp protospacer sequences.}

\item{enzyme}{Character string specifying the Cas9 variant.
Wildtype Cas9 (WT) by default, see details below.}

\item{promoter}{Character string speciyfing promoter used for expressing 
sgRNAs for wildtype Cas9 (must be either "U6" or "T7").
"U6" by default.}

\item{fork}{Set to \code{TRUE} to preserve changes to the R
configuration within the session.}
}
\value{
\strong{getDeepHFScores} returns a data.frame with \code{sequence}
    and \code{score} columns. The DeepHF score takes on a value
    between 0 and 1. A higher score indicates higher knockout efficiency.
}
\description{
Calculate on-target sgRNA activity scores for 
    CRISPR/Cas9-induced knockout using the DeepHF scoring method. Both U6
    and T7 promoters are supported. Three different versions of the SpCas9
    nuclease are supported: wildtype (WT-SpCas9), high-fidelity Cas9
    (SpCas9-HF1) and enhanced Cas9 (eSpCas9). Currently not supported
    on Windows machines.
}
\details{
Input sequences for DeepHF scoring must be 23bpprotospacer
    sequences (20bp spacer sequences + 3bp PAM sequences).
    Only canonical PAM sequences (NGG) are allowed.
    Users can specify for which Cas9 they wish to score sgRNAs
    by using the argument \code{enzyme}: "WT" for Wildtype Cas9 (WT-SpCas9),
    "HF" for high-fidelity Cas9 (SpCas9-HF), or "ESP" for enhanced 
    Cas9 (eSpCas9). For wildtype Cas9, users can also specify the promoter 
    used for expressing sgRNAs using the argument \code{promoter} ("U6" by
    default).
}
\examples{
if (interactive()){
spacer  <- "ATCGATGCTGATGCTAGATA" #20bp
pam     <- "AGG" #3bp 
input   <- paste0(spacer, pam) 

# Wiltype Cas9 using U6 promoter:
results <- getDeepHFScores(input)

# Wiltype Cas9 using T7 promoter:
results <- getDeepHFScores(input, promoter="T7")

#' High-fidelity Cas9:
results <- getDeepHFScores(input, enzyme="HF")

#' Enhanced Cas9:
results <- getDeepHFScores(input, enzyme="ESP")
}

}
\references{
Wang, D., Zhang, C., Wang, B. et al. Optimized CRISPR guide RNA design for
    two high-fidelity Cas9 variants by deep learning.
    Nat Commun 10, 4284 (2019).
    \url{https://doi.org/10.1038/s41467-019-12281-8}
}
\author{
Jean-Philippe Fortin
}
