% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/runCrisprBowtie.R
\name{runCrisprBowtie}
\alias{runCrisprBowtie}
\title{Perform CRISPR gRNA spacer alignment with bowtie}
\usage{
runCrisprBowtie(
  spacers,
  mode = c("protospacer", "spacer"),
  bowtie_index = NULL,
  bsgenome = NULL,
  crisprNuclease = NULL,
  canonical = TRUE,
  ignore_pam = FALSE,
  n_mismatches = 0,
  all_alignments = TRUE,
  n_max_alignments = 1000,
  force_spacer_length = FALSE,
  rna_strict_directionality = TRUE,
  verbose = TRUE
)
}
\arguments{
\item{spacers}{Character vector specifying gRNA spacer sequences.
Sequences must all be of equal length.}

\item{mode}{String specifying which alignment mode should be used:
\code{protospacer} or \code{spacer}. 
For RNA-targeting nucleases such as CasRx, only the 
protospacer mode can be used.}

\item{bowtie_index}{String specifying path to a bowtie index.}

\item{bsgenome}{A \linkS4class{BSgenome} object.
Must be provided if \code{mode} is "spacer".
Ignore}

\item{crisprNuclease}{A \linkS4class{CrisprNuclease} object.}

\item{canonical}{Should only canonical PAM sequences be considered?
TRUE by default.}

\item{ignore_pam}{Should PAM sequences be ignore?
If TRUE, all alignments are returned regardless of PAM tolerance.
FALSE by default.}

\item{n_mismatches}{Integer between 0 and 3 specifying maximum number
of mismatches allowed between spacer sequences and target DNA.
0 by default.}

\item{all_alignments}{Should all possible alignments be returned?
TRUE by default.}

\item{n_max_alignments}{Maximum number of alignments to return if
\code{all_alignments} is FALSE. 1000 by default.}

\item{force_spacer_length}{Should the spacer length be overwritten in the
\code{crisprNuclease} object? FALSE by default.}

\item{rna_strict_directionality}{Should only protospacers found in the 
original direction of the RNA be considered for RNA-targeting
nucleases? TRUE by default.}

\item{verbose}{Should messages be printed to the console? TRUE by default.}
}
\value{
A data.frame of the spacer alignments with the following columns:
   \itemize{
       \item \code{spacer} — string specifying gRNA spacer sequence
       \item \code{protospacer} — string specifying target protospacer sequence
       \item \code{pam} — string specifying target PAM sequence
       \item \code{chr} - string specifying chromosome name
       \item \code{pam_site} - string specifying genomic coordinate of the
             first nucleotide of the PAM sequence.
       \item \code{strand} - string specifying strand ("+" or "-") 
       \item \code{n_mismatches} - integer specifying number of mismatches
             between spacer and protospacer sequences
       \item \code{canonical} - logical indicating whether or not PAM sequence
             is canonical. 
   }
}
\description{
Perform CRISPR gRNA spacer alignment with bowtie.
}
\details{
When \code{mode} is "spacer", spacer sequences are aligned to the
    reference index without appending PAM sequences first. This requires the
    specification of a \linkS4class{BSgenome} object through the argument
    \code{bsgenome} to validate that the aligned spacer sequences are
    adjacent to valid PAM sequences. 
    
    When \code{mode} is "protospacer", sequences are aligned with all
    valid PAM sequences appended (spacer + PAM). The set of valid PAM
    sequences depend on the inputs  \code{canonical} and \code{ignore_pam}.
    This is faster than the "spacer" mode if the number of possible
    PAM sequences is small (e.g. SpCas9).

    For RNA-targeting nucleases, such as RfxCas13d (CasRx), the bowtie
    index should be built on a transcriptome. For such applications,
    only the "protospacer" mode can be used as there is no
    corresponding bsgenome package. The protospacer sequences
    searched in the reference index will be the reverse complement
    of the input spacer sequences.
}
\examples{
fasta <- system.file(package="crisprBowtie", "example/chr1.fa")
outdir <- tempdir()
Rbowtie::bowtie_build(fasta,outdir=outdir, force=TRUE, prefix="tempIndex")
index <- file.path(outdir, "tempIndex")
seqs <- c("GGAAATTCCCCCAGTGGCGC",
          "ACACAGCTGCGGACAGGGCC")
data(SpCas9, package="crisprBase")
results <- runCrisprBowtie(seqs,
                           bowtie_index=index,
                           n_mismatches=2,
                           crisprNuclease=SpCas9)

}
\seealso{
\code{\link{runBowtie}} to map general DNA sequences.
}
\author{
Jean-Philippe Fortin
}
