% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CMap.R
\name{rankSimilarPerturbations}
\alias{rankSimilarPerturbations}
\alias{compareAgainstCMap}
\title{Rank differential expression profile against CMap perturbations by similarity}
\usage{
rankSimilarPerturbations(
  input,
  perturbations,
  method = c("spearman", "pearson", "gsea"),
  geneSize = 150,
  cellLineMean = "auto",
  rankPerCellLine = FALSE,
  threads = 1,
  chunkGiB = 1,
  verbose = FALSE
)
}
\arguments{
\item{input}{\code{Named numeric vector} of differentially expressed genes
whose names are gene identifiers and respective values are a statistic that
represents significance and magnitude of differentially expressed genes
(e.g. t-statistics); or \code{character} of gene symbols composing a gene
set that is tested for enrichment in reference data (only used if
\code{method} includes \code{gsea})}

\item{perturbations}{\code{perturbationChanges} object: CMap perturbations
(check \code{\link{prepareCMapPerturbations}()})}

\item{method}{Character: comparison method (\code{spearman}, \code{pearson}
or \code{gsea}; multiple methods may be selected at once)}

\item{geneSize}{Numeric: number of top up-/down-regulated genes to use as
gene sets to test for enrichment in reference data; if a 2-length numeric
vector, the first index is the number of top up-regulated genes and the
second index is the number of down-regulated genes used to create gene
sets; only used if \code{method} includes \code{gsea} and if \code{input}
is not a gene set}

\item{cellLineMean}{Boolean: add rows with the mean of \code{method} across
cell lines? If \code{cellLineMean = "auto"} (default), rows will be added
when data for more than one cell line is available.}

\item{rankPerCellLine}{Boolean: rank results based on both individual cell
lines and mean scores across cell lines (\code{TRUE}) or based on mean
scores alone (\code{FALSE})? If \code{cellLineMean = FALSE}, individual
cell line conditions are always ranked.}

\item{threads}{Integer: number of parallel threads}

\item{chunkGiB}{Numeric: if second argument is a path to an HDF5 file
(\code{.h5} extension), that file is loaded and processed in chunks of a
given size in gibibytes (GiB); lower values decrease peak RAM usage (see
details below)}

\item{verbose}{Boolean: print additional details?}
}
\value{
Data table with correlation and/or GSEA score results
}
\description{
Compare differential expression results against CMap perturbations.
}
\section{Process data by chunks}{

  If a file path to a valid HDF5 (\code{.h5}) file is provided instead of a
  data matrix, that file can be loaded and processed in chunks of size
  \code{chunkGiB}, resulting in decreased peak memory usage.

  The default value of 1 GiB (1 GiB = 1024^3 bytes) allows loading chunks of ~10000 columns and
  14000 rows (\code{10000 * 14000 * 8 bytes / 1024^3 = 1.04 GiB}).
}

\section{GSEA score}{

  When \code{method = "gsea"}, weighted connectivity scores (WTCS) are
  calculated (\url{https://clue.io/connectopedia/cmap_algorithms}).
}

\examples{
# Example of a differential expression profile
data("diffExprStat")

\dontrun{
# Download and load CMap perturbations to compare with
cellLine <- c("HepG2", "HUH7")
cmapMetadataCompounds <- filterCMapMetadata(
    "cmapMetadata.txt", cellLine=cellLine, timepoint="24 h",
    dosage="5 \u00B5M", perturbationType="Compound")

cmapPerturbationsCompounds <- prepareCMapPerturbations(
    cmapMetadataCompounds, "cmapZscores.gctx", "cmapGeneInfo.txt",
    "cmapCompoundInfo_drugs.txt", loadZscores=TRUE)
}
perturbations <- cmapPerturbationsCompounds

# Rank similar CMap perturbations (by default, Spearman's and Pearson's
# correlation are used, as well as GSEA with the top and bottom 150 genes of
# the differential expression profile used as reference)
rankSimilarPerturbations(diffExprStat, perturbations)

# Rank similar CMap perturbations using only Spearman's correlation
rankSimilarPerturbations(diffExprStat, perturbations, method="spearman")
}
\seealso{
Other functions related with the ranking of CMap perturbations: 
\code{\link{as.table.referenceComparison}()},
\code{\link{filterCMapMetadata}()},
\code{\link{getCMapConditions}()},
\code{\link{getCMapPerturbationTypes}()},
\code{\link{loadCMapData}()},
\code{\link{loadCMapZscores}()},
\code{\link{parseCMapID}()},
\code{\link{plot.perturbationChanges}()},
\code{\link{plot.referenceComparison}()},
\code{\link{plotTargetingDrugsVSsimilarPerturbations}()},
\code{\link{prepareCMapPerturbations}()},
\code{\link{print.similarPerturbations}()}
}
\concept{functions related with the ranking of CMap perturbations}
