#' Sets up overall longtests infrastructure
#'
#' This function is used to set up the environment for running long tests. It
#' calls two helper functions: `setup_bboptions()` and `setup_longtetsts()`.
#'
#' @details The `use_longtests()` function is a wrapper function that calls
#' `setup_bboptions()` and `setup_longtetsts()`. The `setup_bboptions()`
#' function checks if the .BBSoptions file exists and creates it if it doesn't.
#' It then checks the contents of the .BBSoptions file and adds or modifies the
#' 'RunLongTests: TRUE' line as needed. The `setup_longtetsts()` function
#' creates a 'longtests/testthat' directory if it doesn't exist and copies the
#' 'tests/testthat.R' file into it. If the 'tests/testthat.R' file doesn't
#' exist, it creates a new one with default content.
#'
#' @return This function does not return a value. It is used for its side
#'   effects of setting up the environment for running long tests.
#' @export
#' @examples
#' # Create the longtests directory and .BBSoptions file
#' use_longtests()
#'
#' # Remove the longtests directory and .BBSoptions file
#' unlink(file.path(".BBSoptions"), recursive = TRUE, force = TRUE)
#' unlink(file.path("longtests"), recursive = TRUE, force = TRUE)
use_longtests <- function() {
  setup_bboptions()
  setup_longtetsts()
}

#' @noRd
#' @keywords internal
setup_longtetsts <- function() {
  path <- file.path("longtests", "testthat")
  dir.create(path, recursive = TRUE, showWarnings = FALSE)
  if (file.exists(file.path("tests", "testthat.R"))) {
    file.copy(
      file.path("tests", "testthat.R"),
      file.path("longtests", "testthat.R"),
      overwrite = TRUE
    )
  } else {
    '
      # This file is part of the standard setup for testthat.
      # It is recommended that you do not modify it.
      #
      # Where should you do additional test configuration?
      # Learn more about the roles of various files in:
      # * https://r-pkgs.org/testing-design.html#sec-tests-files-overview
      # * https://testthat.r-lib.org/articles/special-files.html

      library(testthat)
      library(biocroxytest)

      test_check("biocroxytest")
    ' |> writeLines(file.path("longtests", "testthat.R"))
  }

  cli::cli_inform(
    c(
      "v" = "The {cli::col_blue('longtests')} directory has been created.",
      "v" = "Adding {cli::col_blue('testthat.R')} to the {cli::col_blue('longtests')} directory."
    )
  )
}

#' @noRd
#' @keywords internal
setup_bboptions <- function() {
  # Check if .BBSoptions exists
  if (!file.exists(".BBSoptions")) {
    file.create(".BBSoptions")
    cli::cli_inform(
      c(
        "v" = "The {cli::col_blue('.BBSoptions')} file has been created.",
        "v" = "Adding {cli::col_blue('RunLongTests: TRUE')} to the {cli::col_blue('.BBSoptions')} file."
      )
    )

    writeLines(
      c("# Generated by biocroxytest: do not edit by hand!", "RunLongTests: TRUE"),
      ".BBSoptions"
    )
  } else {
    lines <- readLines(".BBSoptions")
    if (length(lines) == 0) { lines = "# Generated by biocroxytest: do not edit by hand!" }
    if (any(stringr::str_detect(lines, "RunLongTests: TRUE"))) {
      cli::cli_inform(
        c(
          "i" = "The {cli::col_blue('.BBSoptions')} file contains {cli::col_blue('RunLongTests: TRUE')}",
          "v" = "Nothing to do."
        )
      )
    }

    if (!any(stringr::str_detect(lines, "RunLongTests:"))) {
      cli::cli_inform(
        c(
          "i" = "The {cli::col_blue('.BBSoptions')} file does not contain {cli::col_blue('RunLongTests')}",
          "v" = "Adding {cli::col_blue('RunLongTests: TRUE')} to the {cli::col_blue('.BBSoptions')} file"
        )
      )
      writeLines(c(lines, "RunLongTests: TRUE"), ".BBSoptions")
    }

    if (any(stringr::str_detect(lines, "RunLongTests: FALSE"))) {
      cli::cli_inform(
        c(
          "i" = "The {cli::col_blue('.BBSoptions')} file contains {cli::col_blue('RunLongTests: FALSE')}",
          "v" = "Changing {cli::col_blue('RunLongTests: FALSE')} to {cli::col_blue('RunLongTests: TRUE')}"
        )
      )
      lines <- stringr::str_replace(lines, "RunLongTests: FALSE", "RunLongTests: TRUE")
      writeLines(lines, ".BBSoptions")
    }
  }

  invisible(TRUE)
}
