% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/maxChangeOverview.R
\name{maxChangeOverview}
\alias{maxChangeOverview}
\title{Gives an overview of the susceptibility to max change
attacks, for each cell type, for a given list of modifications.}
\usage{
maxChangeOverview(
  exprs,
  clusters,
  classifier,
  exclGenes = c(),
  genes = c(),
  modifications = list(c("perc1"), c("perc99")),
  advMethod = "perc99",
  advFixedValue = 3,
  advFct = NULL,
  maxSplitSize = 100,
  argForClassif = "data.frame",
  argForModif = "data.frame",
  verbose = FALSE
)
}
\arguments{
\item{exprs}{DelayedMatrix of numeric RNA expression, cells are rows and genes
are columns - or a SingleCellExperiment object, a matrix or a data.frame. By default,
these are converted to a data.frame to increase speed performance during modifications.
However, this conversion can consume a significant amount of memory, see 'argForModif'
argument for options.}

\item{clusters}{a character vector of the clusters to which the cells belong}

\item{classifier}{a classifier in the suitable format.
A classifier function should be formated as follow:
classifier = function(expr, clusters, target){
# Making the classification
c("cell type", score)
}
\code{score} should be numeric between 0 and 1, 1 being the highest confidance
into the cell type classification.
The matrix \code{expr} contains RNA expression values, the vector \code{clusters}
consists of the cluster IDs for each cell in \code{expr}, and \code{target} is the
ID of the cluster for which we want to have a classification.
The function returns a vector with the classification result, and a score.}

\item{exclGenes}{a character vector of genes to exclude from the analysis}

\item{genes}{a character vector of genes in case you want to limit the
analysis on a subset of genes}

\item{modifications}{the list of the modifications to study}

\item{advMethod}{the name of the method to use}

\item{advFixedValue}{the numeric value to use in case of
advMethod=\code{fixed}}

\item{advFct}{the function to use in case advMethod
belongs to the following list: \code{full_row_fct}, \code{target_row_fct},
\code{target_matrix_fct}, \code{full_matrix_fct}}

\item{maxSplitSize}{max size of dichotomic slices.}

\item{argForClassif}{the type of the first argument to feed to the
classifier function. 'data.frame' by default, can be 'SingleCellExperiment'
or 'DelayedMatrix'.}

\item{argForModif}{type of matrix during for the modification, 'data.frame'
by default. Can be 'DelayedMatrix', which needs less memory but is slower.}

\item{verbose}{logical, set to TRUE to activate verbose mode}
}
\value{
a DataFrame storing the number of possible max change attacks
for each cell type and each modification.
}
\description{
Gives an overview of the susceptibility to max change
attacks, for each cell type, for a given list of modifications.
}
\details{
Running the advMaxChange function for each cell type
to see which ones are more vulerable can take a long time. The
aim of the maxChangeOverview function is to make this process faster.
It uses a default value of 100 for the 'maxSplitSize' parameter. So,
the dichotomic process of the advMaxChange function stops as soon
as the fold length is lower than 100. You can have more accurate
results with maxSplitSize=1, but it will take longer.
This function aims also to run the advMaxChange for several given
modifications. You can specify a list of modifications as so - each
item of the list should be 1 or 2 length size.
The 1 length vector must contain the prerecorded modifications, 'perc1'
or 'perc99'.
The 2 length vector must have as first item:
\itemize{
\item 'fixed', in this case the second item should be the value to be
replaced by.
\item 'full_row_fct', 'target_row_fct', 'target_matrix_fct' or
'full_matrix_fct'. In this case the second item should be a function.
Let's say we want to analysis the susceptibility to max change attack
for 3 modifications: "perc1", the modification of each value of the
cluster by 1000, and a custom modification stored inside a function myFct.
Then the 'modification' parameter should be:
my_modifications = list(c("perc1"),
c("fixed", 1000),
c("full_matrix_fct", myFct))
}

The function returns a dataframe with the number of genes of the max
change attack for each modification in columns, for each cell type in rows.
}
\examples{
library(DelayedArray)

MyClassifier <- function(expr, clusters, target) {
   c("T cell", 0.9)
}
rna_expression <- DelayedArray(data.frame(CD4=c(0,0,0,0), CD8A=c(1,1,1,1),
     CD8B=c(2,2,3,3)))
genes <- c("CD4", "CD8A")
clusters_id <- c("B cell","B cell","T cell","T cell")

maxChangeOverview(rna_expression, clusters_id,
MyClassifier, modifications = list(c("perc1"), c("perc99")))

myModif = function(x, y){
   return(sample(1:10,1))
}

my_modifications = list(c("perc1"),
                        c("fixed", 1000),
                        c("full_matrix_fct", myModif))
maxChangeOverview(rna_expression, clusters_id,
 MyClassifier, modifications = my_modifications)
}
