\name{ri_data_extract}
\alias{ri_data_extract}
\title{Extract peak data from a RI file}
\usage{
ri_data_extract(RIfile, massValues, timeRange, useRT = FALSE, ...)
}
\arguments{
\item{RIfile}{A path to a RI file format (binary or text).}

\item{massValues}{A numeric vector representing m/z values. The values will be converted
to integers internally.}

\item{timeRange}{A numeric vector of even length, or, a two-column matrix. This parameter
represents the lower and upper time limits. See details below.}

\item{useRT}{Logical. If \code{TRUE}, the time range is in seconds, otherwise if \code{FALSE}
(default), the time range is in retention time index units (\code{TRUE}).}

\item{...}{Extra parameters passed to internal functions. Currently, only the \code{columns}
parameter is used for text-format RI files. This, however, should \strong{not} be necessary.
See also \code{\link{text2bin}}.}
}
\value{
A four column matrix with column names described below. Each row represents a peak
in the specified time range. Potentially, there could be zero rows if no peaks are found.
\itemize{
\item \code{RI} retention time index of the peak.
\item \code{RT} retention time.
\item \code{Intensity} peak height intensity
\item \code{mz} the \emph{m/z} of the peak
}
}
\description{
A convenient function to extract peak data from a RI file. This function is mostly
intended for debugging purposes, for example, when a given peak is not found within
a expected time range (which could be because a faulty retention time correction).
The arguments of this function are similar as those of the \code{\link{ncdf4_data_extract}} function.
}
\details{
The function takes a RI file generated by \code{\link{TargetSearch}} (binary or text works)
and extracts all the peaks detected of given \emph{m/z} ion traces in a given time range.
The time range can be in seconds or arbitrary retention time index units.

The parameter \code{timeRange} can be either a numeric vector of even length or a
two-column matrix. If it is a matrix, the first column represent the lower bounds
while the second the upper ones. If it is a vector, it must have at least two elements,
which in this case represent the lower and upper limits, or if longer, it will be coerced to matrix
internally (\strong{Important}: the values are filled by \emph{columns}). In any case,
the number of rows of the (coerced) matrix must divide the length of the \code{massValues}
parameter, such that each row corresponds to a time range for each \emph{m/z} value.
This is done so the ranges can be recycled.

The output is simply a four column matrix in which a row is a peak of a given ion
trace at a retention time (index). The output is always a matrix even if no peaks
are found, in which case the number of rows is zero.
}
\note{
This function is intended to be used internally or by advances users. It can be used
for debugging when \code{\link{TargetSearch}} fails to find a peak.

See also the function \code{\link{FindAllPeaks}} which offers a similar functionality
but with different input parameters.
}
\examples{
require(TargetSearchData)
path <- tsd_data_path()
rifile <- tsd_rifiles()[1]

# extract peak data for m/z 116 and 117 in time range 180 - 220 seconds
data <- ri_data_extract(rifile, c(116, 117), c(180, 220), useRT=TRUE)

# same but using Retention Time Index
data <- ri_data_extract(rifile, c(116, 117), c(205000, 228000), useRT=FALSE)

# different time ranges for each m/z
trange <- c(180, 240, 280, 190, 250, 290)
data <- ri_data_extract(rifile, c(116, 144, 147), trange, useRT=TRUE)

# Note: the range definition above is equivalent to
trange <- cbind(c(180,240,280), c(190, 250, 290))

}
\seealso{
    \code{\link{text2bin}},
    \code{\link{ncdf4_data_extract}},
    \code{\link{FindAllPeaks}},
    \code{\link{FindPeaks}}
}
\author{
Alvaro Cuadros-Inostroza
}
% vim: set ts=4 sw=4 et:
