\name{baseline}
\alias{baseline}
\title{Baseline correction - wrapper function}
\description{
    This function perform baseline correction by wrapping around the methods
    implemented on \code{\link{baselineCorrection}} and \code{\link{baselineCorrectionQuant}}.
}
\usage{
    baseline(ncdf, bsline_method = c('classic', 'quantiles', 'none'), ...)
}
\arguments{
  \item{ncdf}{A list containing the raw chromatogram data. The list can be generated by
  \code{\link{peakCDFextraction}}. Expected elements are \code{"Peaks"} which is matrix
  of intensities where the rows are retention times and columns are mass traces, \code{"Time"}
  which is a vector of retention time in seconds.}
  \item{bsline_method}{A string to select the baseline retention method. Options
  are \code{"classic"} which implements Chang's method (the old or classic TargetSearch method),
  \code{"quantiles"} the quantiles based method, and \code{"none"} which does nothing
  (returns the same input).}
  \item{\dots}{Extra parameters to be passed to \code{\link{baselineCorrection}} or
  \code{\link{baselineCorrectionQuant}}}.
}
\details{
This is a wrapper function around the different baseline correction algorithms. It
is not intended to be executed by the average user. Please refer to the respective
man pages for details.}
\value{
  Returns a list with the same elements as the input, but the element \code{"Peaks"}
  containing baseline corrected values.
}
\author{Alvaro Cuadros-Inostroza}
\seealso{ \code{\link{RIcorrect}}, \code{\link{baselineCorrection}}, \code{\link{baselineCorrectionQuant}} }
\examples{
  # get a random sample CDF from TargetSearchData
  require(TargetSearchData)
  cdffile <- sample(tsd_cdffiles(), 1)
  pdata <- peakCDFextraction(cdffile)

  # restrict mass range to reduce computing time (not needed for
  # actual data)
  pdata$Peaks <- pdata$Peaks[, 1:10] ; pdata$massRange <- c(85, 94)

  # make a fake baseline as constant + noise (the CDF files have been
  # already baseline corrected by the vendor software).
  nscans <- length(pdata$Time)
  noise <- as.integer(1000 + rnorm(nscans, sd=5))
  pdata$Peaks <- pdata$Peaks + noise

  # use Classic and Quantile methods for baseline correction (def parameters)
  pdata_c <- baseline(pdata, 'classic')
  # use Quantile method
  pdata_q <- baseline(pdata, 'quantiles')

  # plot function to compare traces
  plfun <- function(p, q, k, n, titl) {
    plot(p$Time, p$Peaks[, k], col='blue', type='l', xlab='time', ylab='intensity')
    lines(q$Time, q$Peaks[, k] - n, col='red')
    legend('topleft', c('corrected', 'original'), col=c('blue', 'red'), lty=1, lwd=2)
    title(paste('method:', titl))
  }

  op <- par(mfrow=c(2,2))
    plfun(pdata_c, pdata, 1, noise, 'classic')
    plfun(pdata_q, pdata, 1, noise, 'quantile')
    plfun(pdata_c, pdata, 7, noise, 'classic')
    plfun(pdata_q, pdata, 7, noise, 'quantile')
  par(op)
}
