% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Coverage.R
\name{Coverage}
\alias{Coverage}
\alias{getCoverage}
\alias{getCoverage_DF}
\alias{getCoverageRegions}
\alias{getCoverageBins}
\title{Calls SpliceWiz's C++ function to retrieve coverage from a COV file}
\usage{
getCoverage(file, seqname = "", start = 0, end = 0, strand = c("*", "+", "-"))

getCoverage_DF(
  file,
  seqname = "",
  start = 0,
  end = 0,
  strand = c("*", "+", "-")
)

getCoverageRegions(
  file,
  regions,
  strandMode = c("unstranded", "forward", "reverse")
)

getCoverageBins(
  file,
  region,
  bins = 2000,
  strandMode = c("unstranded", "forward", "reverse"),
  bin_size
)
}
\arguments{
\item{file}{(Required) The file name of the COV file}

\item{seqname}{(Required for \code{getCoverage_DF}) A string denoting the
chromosome name. If left blank in \code{getCoverage}, retrieves RLEList
containing coverage of the entire file.}

\item{start, end}{1-based genomic coordinates. If \code{start = 0} and
\code{end = 0}, will retrieve RLE of specified chromosome.}

\item{strand}{Either "*", "+", or "-"}

\item{regions}{A GRanges object for a set of regions to obtain mean / total
coverage from the given COV file.}

\item{strandMode}{The stranded-ness of the RNA-seq experiment. "unstranded"
means that an unstranded protocol was used. Stranded protocols can be
either "forward", where the first read is the same strand as the
expressed transcript, or "reverse" where the second strand is the same
strand as the expressed transcript.}

\item{region}{In \code{getCoverageBins}, a single query region as a GRanges object}

\item{bins}{In \code{getCoverageBins}, the number of bins to divide the given
\code{region}. If \code{bin_size} is given, overrides this parameter}

\item{bin_size}{In \code{getCoverageBins}, the number of nucleotides per bin}
}
\value{
For \code{getCoverage}: If seqname is left as "", returns an RLEList of the
whole BAM file, with each RLE in the list containing coverage data for
one chromosome. Otherwise, returns an RLE containing coverage data for
the requested genomic region

For \code{getCoverage_DF}: Returns a two-column data frame, with the first column
\code{coordinate} denoting genomic coordinate, and the second column \code{value}
containing the coverage depth for each coordinate nucleotide.

For \code{getCoverageRegions}: Returns a GRanges object with an extra metacolumn:
\code{cov_mean}, which gives the mean coverage of each of the given ranges.

For \code{getCoverageBins}: Returns a GRanges object which spans the given
\code{region}, divided by the number of \code{bins} or by width as given by
\code{bin_size}. Mean coverage in each bin is calculated (returned by the
\code{cov_mean} metadata column). This function is useful for retrieving
coverage of a large region for visualisation, especially when the
size of the region vastly exceeds the width of the figure.
}
\description{
This function returns an RLE / RLEList or data.frame
containing coverage data from the given COV file\cr\cr
COV files are generated by SpliceWiz's \link{processBAM} and \link{BAM2COV} functions.
It records alignment coverage for each nucleotide in the given BAM file.
It stores this data in "COV" format, which is an indexed BGZF-compressed
format specialised for the storage of unstranded and stranded alignment
coverage in RNA sequencing.\cr\cr
Unlike BigWig files, COV files store coverage for both positive and negative
strands.\cr\cr
These functions retrieves coverage data from the specified COV file. They
are computationally efficient as they utilise random-access to rapidly
search for the requested data from the COV file.\cr\cr
}
\section{Functions}{
\itemize{
\item \code{getCoverage()}: Retrieves alignment coverage as an RLE or RLElist

\item \code{getCoverage_DF()}: Retrieves alignment coverage as a data.frame

\item \code{getCoverageRegions()}: Retrieves total and mean coverage of a GRanges object
from a COV file

\item \code{getCoverageBins()}: Retrieves coverage of a single region from a COV file,
binned by the given number of bins or bin_size

}}
\examples{
se <- SpliceWiz_example_NxtSE()

cov_file <- covfile(se)[1]

# Retrieve Coverage as RLE

cov <- getCoverage(cov_file, seqname = "chrZ",
  start = 10000, end = 20000,
  strand = "*"
)

# Retrieve Coverage as data.frame

cov.df <- getCoverage_DF(cov_file, seqname = "chrZ",
  start = 10000, end = 20000,
  strand = "*"
)

# Retrieve mean coverage of 100-nt window regions as defined
# in a GRanges object:

gr <- GenomicRanges::GRanges(
    seqnames = "chrZ",
    ranges = IRanges::IRanges(
        start = seq(1, 99901, by = 100),
        end = seq(100, 100000, by = 100)
    ), strand = "-"
)

gr.unstranded <- getCoverageRegions(cov_file,
    regions = gr,
    strandMode = "unstranded"
)

gr.stranded <- getCoverageRegions(cov_file,
    regions = gr,
    strandMode = "reverse"
)

# Retrieve binned coverage of a large region

gr.fetch <- getCoverageBins(
    cov_file,
    region = GenomicRanges::GRanges(seqnames = "chrZ",
        ranges = IRanges::IRanges(start = 100, end = 100000),
        strand = "*"
    ),
    bins = 2000
)

# Plot coverage using ggplot:

require(ggplot2)

ggplot(cov.df, aes(x = coordinate, y = value)) +
    geom_line() + theme_white

ggplot(as.data.frame(gr.unstranded),
    aes(x = (start + end) / 2, y = cov_mean)) +
    geom_line() + theme_white

ggplot(as.data.frame(gr.fetch), 
    aes(x = (start + end)/2, y = cov_mean)) +
    geom_line() + theme_white

# Export COV data as BigWig

cov_whole <- getCoverage(cov_file)
bw_file <- file.path(tempdir(), "sample.bw")
rtracklayer::export(cov_whole, bw_file, "bw")
}
