% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/deprofile.R
\name{deprofile}
\alias{deprofile}
\alias{deprofile.scan}
\alias{deprofile.fwhm}
\alias{deprofile.localMax}
\alias{deprofile.spline}
\title{De-profile a high-resolution MS scan in profile mode.}
\usage{
deprofile.scan(scan, noise = NA, method = "deprofile.fwhm", 
					colnames = TRUE, ...)

				deprofile(df, noise, method, ...)

				deprofile.fwhm(df, noise = NA, cut = 0.5)

				deprofile.localMax(df, noise = NA)

				deprofile.spline(df, noise=NA, minPts = 5, step = 0.00001)
}
\arguments{
\item{df}{A dataframe with at least the columns \code{mz} and \code{int} to perform deprofiling on.}

\item{noise}{The noise cutoff. A peak is not included if the maximum stick intensity of the peak
is below the noise cutoff.}

\item{method}{"deprofile.fwhm" for full-width half-maximum or "deprofile.localMax" for
local maximum.}

\item{...}{Arguments to the workhorse functions \code{deprofile.fwhm} etc.}

\item{scan}{A matrix with 2 columns for m/z and intensity; from profile-mode high-resolution data. Corresponds
to the spectra obtained with xcms::getScan or mzR::peaks.}

\item{colnames}{For deprofile.scan: return matrix with column names (xcms-style, 
\code{TRUE}, default) or plain (mzR-style, \code{FALSE}).}

\item{cut}{A parameter for \code{deprofile.fwhm} indicating where the peak flanks should be taken. Standard is 0.5
(as the algorithm name says, at half maximum.) Setting \code{cut = 0.75} would instead determine the peak
width at 3/4 maximum, which might give a better accuracy for merged peaks, but could be less accurate
if too few data points are present.}

\item{minPts}{The minimal points count in a peak to build a spline; for peaks with less
points the local maximum will be used instead. Note: The minimum value
is 4!}

\item{step}{The interpolation step for the calculated spline, which limits the maximum 
precision which can be achieved.}
}
\value{
\code{deprofile.scan}: a matrix with 2 columns for m/z and intensity
}
\description{
The \code{deprofile} functions convert profile-mode high-resolution input data to "centroid"-mode
data amenable to i.e. centWave. This is done using full-width, half-height algorithm, spline
algorithm or local maximum method.
}
\details{
The \code{deprofile.fwhm} method is basically an R-semantic version of the "Exact Mass" m/z deprofiler
from MZmine. It takes the center between the m/z values at half-maximum intensity for the exact peak mass.
The logic is stolen verbatim from the Java MZmine algorithm, but it has been rewritten to use the fast
R vector operations instead of loops wherever possible. It's slower than the Java implementation, but 
still decently fast IMO. Using matrices instead of the data frame would be more memory-efficient
and also faster, probably.

The \code{deprofile.localMax} method uses local maxima and is probably the same used by e.g. Xcalibur.
For well-formed peaks, "deprofile.fwhm" gives more accurate mass results; for some applications,
\code{deprofile.localMax} might be better (e.g. for fine isotopic structure peaks which are
not separated by a valley and also not at half maximum.) For MS2 peaks, which have no isotopes,
\code{deprofile.fwhm} is probably the better choice generally.

\code{deprofile.spline} calculates the mass using a cubic spline, as the HiRes peak detection
in OpenMS does.

The word "centroid" is used for convenience to denote not-profile-mode data.
The data points are NOT mathematical centroids; we would like to have a better word for it.

The \code{noise} parameter was only included for completeness, I personally don't use it.

\code{deprofile.fwhm} and \code{deprofile.localMax} are the workhorses; 
\code{deprofile.scan} takes a 2-column scan as input.
\code{deprofile} dispatches the call to the appropriate worker method.
}
\note{
Known limitations: If the absolute leftmost stick or the absolute rightmost stick in
				a scan are maxima, they will be discarded! However, I don't think this will
				ever present a practical problem.
}
\examples{
\dontrun{
mzrFile <- openMSfile("myfile.mzML")
acqNo <- xraw@acquisitionNum[[50]]
scan.mzML.profile <- mzR::peaks(mzrFile, acqNo)
scan.mzML <- deprofile.scan(scan.mzML.profile) 
close(mzrFile)
}

}
\references{
mzMine source code \href{http://sourceforge.net/svn/?group_id=139835}{http://sourceforge.net/svn/?group_id=139835}
}
\author{
Michael Stravs, Eawag <michael.stravs@eawag.ch>
}
