% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/readQFeaturesFromDIANN.R
\name{readQFeaturesFromDIANN}
\alias{readQFeaturesFromDIANN}
\title{Read DIA-NN output as a QFeatures objects}
\usage{
readQFeaturesFromDIANN(
  assayData,
  colData = NULL,
  quantCols = "Ms1.Area",
  runCol = "File.Name",
  multiplexing = c("none", "mTRAQ"),
  extractedData = NULL,
  ecol = NULL,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{assayData}{A \code{data.frame}, or any object that can be coerced
into a \code{data.frame}, holding the quantitative assay. For
\code{readSummarizedExperiment()}, this can also be a
\code{character(1)} pointing to a filename. This \code{data.frame} is
typically generated by an identification and quantification
software, such as Sage, Proteome Discoverer, MaxQuant, ...}

\item{colData}{A \code{data.frame} (or any object that can be coerced
to a \code{data.frame}) containing sample/column annotations,
including \code{quantCols} and \code{runCol} (see details).}

\item{quantCols}{A \code{numeric()}, \code{logical()} or \code{character()}
defining the columns of the \code{assayData} that contain the
quantitative data. This information can also be defined in
\code{colData} (see details).}

\item{runCol}{For the multi-set case, a \code{numeric(1)} or
\code{character(1)} pointing to the column of \code{assayData} (and
\code{colData}, is set) that contains the runs/batches. Make sure
that the column name in both tables are identical and
syntactically valid (if you supply a \code{character}) or have the
same index (if you supply a \code{numeric}). Note that characters
are converted to syntactically valid names using \code{make.names}}

\item{multiplexing}{A \code{character(1)} indicating the type of
multiplexing used in the experiment. One of \code{"none"} (default,
for label-free experiments) or \code{"mTRAQ"} (for plexDIA
experiments).}

\item{extractedData}{A \code{data.frame} or any object that can be
coerced to a \code{data.frame} that contains the data from the
\verb{*_ms1_extracted.tsv} file generated by DIA-NN. This argument
is optional and is currently only applicable for mTRAQ
multiplexed experiments where DIA-NN was run using the
\code{plexdia} module (see references).}

\item{ecol}{Same as \code{quantCols}. Available for backwards
compatibility. Default is \code{NULL}. If both \code{ecol} and \code{colData}
are set, an error is thrown.}

\item{verbose}{A \code{logical(1)} indicating whether the progress of
the data reading and formatting should be printed to the
console. Default is \code{TRUE}.}

\item{...}{Further arguments passed to \code{\link[=readQFeatures]{readQFeatures()}}.}
}
\value{
An instance of class \code{QFeatures}. The quantiative data of
each acquisition run is stored in a separate set as a
\code{SummarizedExperiment} object.
}
\description{
This function takes the \code{Report.tsv} output files from DIA-NN and
converts them into a multi-set \code{QFeatures} object. It is a wrapper
around \code{\link[=readQFeatures]{readQFeatures()}} with default parameters set to match
DIA-NN label-free and plexDIA report files: default \code{runCol} is
\code{"File.Name"} and default quantCols\code{is}"Ms1.Area"`.
}
\examples{

x <- read.delim(MsDataHub::benchmarkingDIA.tsv())
x[["File.Name"]] <- x[["Run"]]

#################################
## Label-free multi-set case

## using default arguments
readQFeaturesFromDIANN(x)

## use the precursor identifier as assay rownames
readQFeaturesFromDIANN(x, fnames = "Precursor.Id") |>
    rownames()

## with a colData (and default arguments)
cd <- data.frame(sampleInfo = LETTERS[1:24],
                 quantCols = "Ms1.Area",
                 runCol = unique(x[["File.Name"]]))
readQFeaturesFromDIANN(x, colData = cd)

#################################
## mTRAQ multi-set case

x2 <- read.delim(MsDataHub::Report.Derks2022.plexDIA.tsv())
x2[["File.Name"]] <- x2[["Run"]]
readQFeaturesFromDIANN(x2, multiplexing = "mTRAQ")
}
\references{
Derks, Jason, Andrew Leduc, Georg Wallmann, R. Gray Huffman,
Matthew Willetts, Saad Khan, Harrison Specht, Markus Ralser,
Vadim Demichev, and Nikolai Slavov. 2022. "Increasing the
Throughput of Sensitive Proteomics by plexDIA." Nature
Biotechnology, July.
\href{http://dx.doi.org/10.1038/s41587-022-01389-w}{Link to article}
}
\seealso{
\itemize{
\item The \code{QFeatures} (see \code{\link[=QFeatures]{QFeatures()}}) class to read about how to
manipulate the resulting \code{QFeatures} object.
\item The \code{\link[=readQFeatures]{readQFeatures()}} function which this one depends on.
}
}
\author{
Laurent Gatto, Christophe Vanderaa
}
