#' Calculate motif enrichment in a set of sequences
#'
#' \code{motif_enrichment()} calculates motif enrichment relative to a set of
#' background sequences using Analysis of Motif Enrichment (AME) from
#' \link[memes]{memes-package}.
#'
#' @param peak_input Either a path to the narrowPeak file or a GRanges peak
#' object generated by \code{read_peak_file()}.
#' @param motif An object of class \code{universalmotif}.
#' @param genome_build The genome build that the peak sequences should be
#' derived from.
#' @param out_dir Location to save the 0-order background file along with the
#' AME output files.
#' @param verbose A logical indicating whether to print verbose messages while
#' running the function. (default = FALSE)
#' @inheritParams memes::runAme
#' @inheritDotParams memes::runAme -input -control -outdir -database
#' 
#' @importFrom BSgenome getSeq
#' @importFrom memes runAme
#' @importFrom utils read.table
#'
#' @returns A list containing a AME results data frame and a numeric referring
#' to the proportion of peaks with a motif.
#'
#' @examples
#' if (memes::meme_is_installed()) {
#'     data("CTCF_TIP_peaks", package = "MotifPeeker")
#'     data("motif_MA1102.3", package = "MotifPeeker")
#' 
#'     res <- motif_enrichment(
#'         peak_input = CTCF_TIP_peaks,
#'         motif = motif_MA1102.3,
#'         genome_build =
#'             BSgenome.Hsapiens.UCSC.hg38::BSgenome.Hsapiens.UCSC.hg38,
#'         
#'     )
#'     print(res)
#' }
#'
#' @seealso \code{\link[memes]{runAme}}
#' 
#' @export
motif_enrichment <- function(peak_input,
                            motif,
                            genome_build,
                            out_dir = tempdir(),
                            verbose = FALSE,
                            meme_path = NULL,
                            ...) {
    if (!dir.exists(out_dir)) dir.create(out_dir, recursive = TRUE)
    
    empty_result <- list(tp = c(0, 0),
                        fp = c(NA, NA),
                        positive_peaks = NA)
    
    ## Handle empty input
    if (length(peak_input) == 0) {
        messager("No peaks detected in input for motif_enrichment.",
                "Returning empty result.", v = verbose)
        return(empty_result)
    }
    
    peaks <- peak_input # Backwards compatibility
    peak_sequences <- BSgenome::getSeq(genome_build, peak_input)
    
    ## Generate 0-order background model from the input sequences
    # bfile <- markov_background_model(sequences = peak_sequences,
    #                                 out_dir = out_dir)
    ame_out <- memes::runAme(peak_sequences,
                            database = list(motif),
                            outdir = out_dir,
                            meme_path = meme_path)
    
    ## Handle zero enrichment
    if (is.null(ame_out)) {
        messager("No peaks were enriched for the input motif.",
                "Returning empty result.", v = verbose)
        return(empty_result)
    }
    
    ## Read output
    seq_path <- file.path(out_dir, "sequences.tsv")
    seq <- utils::read.table(seq_path, header = TRUE)
    cleaned_ids <- seq$seq_ID[!grepl("_shuf_", seq$seq_ID)]
    
    return(
        list(
            tp = c(ame_out$tp, ame_out$tp_percent),
            fp = c(ame_out$fp, ame_out$fp_percent),
            positive_peaks = peaks[cleaned_ids]
        )
    )
}
