% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SimpleCOMPASS.R
\name{SimpleCOMPASS}
\alias{SimpleCOMPASS}
\title{Fit the discrete COMPASS Model}
\usage{
SimpleCOMPASS(
  n_s,
  n_u,
  meta,
  individual_id,
  iterations = 10000,
  replications = 8,
  verbose = TRUE,
  seed = 100
)
}
\arguments{
\item{n_s}{The cell counts for stimulated cells.}

\item{n_u}{The cell counts for unstimulated cells.}

\item{meta}{A \code{data.frame} of metadata, describing the individuals
in the experiment. Each row in \code{meta} should correspond to a row
in \code{data}. There should be one row for each subject;
i.e., one row for each element of \code{n_s} and \code{n_u}.}

\item{individual_id}{The name of the vector in \code{meta} that denotes the
individuals from which samples were drawn.}

\item{iterations}{The number of iterations (per 'replication') to perform.}

\item{replications}{The number of 'replications' to perform. In order to
conserve memory, we only keep the model estimates from the last replication.}

\item{verbose}{Boolean; if \code{TRUE} we output progress information.}

\item{seed}{A seed for the random number generator. Defaults to 100.}
}
\value{
A \code{list} with class \code{COMPASSResult} with two components,
  the \code{fit} containing parameter estimates and parameter acceptance
  rates, and \code{data} containing the generated data used as input for
  the model.
}
\description{
This function fits the \code{COMPASS} model from a user-provided set of
stimulated / unstimulated matrices. See the NOTE for important details.
}
\note{
n_s and n_u counts matrices should contain ALL 2^M possible combinations of markers, even if they are 0 for some combinations. The code expects the marker combinations to be named in the following way:
\code{"M1&M2&!M3"} means the combination represents cells expressing marker "M1" and "M2" and not "M3". For 3 markers, there should be 8 such combinations, such that n_s and n_u have 8 columns.
}
\examples{
 set.seed(123)
n <- 10 ## number of subjects
k <- 3 ## number of markers

## generate some sample data
iid_vec <- paste0("iid_", 1:n) # Subject id
data <- replicate(2*n, {
nrow <- round(runif(1) * 1E4 + 1000)
ncol <- k
vals <- rexp( nrow * ncol, runif(1, 1E-5, 1E-3) )
vals[ vals < 2000 ] <- 0
output <- matrix(vals, nrow, ncol)
output <- output[ apply(output, 1, sum) > 0, ]
colnames(output) <- paste0("M", 1:k)
return(output)
})

meta <- cbind(iid=iid_vec, data.frame(trt=rep( c("Control", "Treatment"), each=n/2 )))

## generate counts for n_s, n_u
n_s <- CellCounts( data[1:n], Combinations(k) )
n_u <- CellCounts( data[(n+1):(2*n)], Combinations(k) )
rownames(n_s) = unique(meta$iid)
rownames(n_u) = rownames(n_s)
## A smaller number of iterations is used here for running speed;
## prefer using more iterations for a real fit
scr = SimpleCOMPASS(n_s, n_u, meta, "iid", iterations=1000)
}
