% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/COMPASSContainer.R
\name{COMPASSContainer}
\alias{COMPASSContainer}
\title{Generate the Data Object used by COMPASS}
\usage{
COMPASSContainer(
  data,
  counts,
  meta,
  individual_id,
  sample_id,
  countFilterThreshold = 0
)
}
\arguments{
\item{data}{A list of matrices. Each matrix \code{M_i} is made up of
\code{N_i} cells by \code{K} markers; for example, it could be the
intensity information from an intracellular cytokine experiment.
Each element of the list should be named; this name denotes which
sample the cell intensities were measured from.}

\item{counts}{A named integer vector of the cell counts(of the parent population) for each
sample in \code{data}.}

\item{meta}{A \code{data.frame} of metadata, describing the individuals
in the experiment. Each row in \code{meta} should correspond to a row
in \code{data}. There should be one row for each sample;
i.e., one row for each element of \code{data}.}

\item{individual_id}{The name of the vector in \code{meta} that denotes the
individuals from which samples were drawn. In this sense an individual equates to a single subject, or person.}

\item{sample_id}{The name of the vector in \code{meta} that denotes the samples. The sample_id identifies a combination of a subject with visit (if any), cell subset measured (e.g. CD4), and stimulation. 
This vector should contain all of the names in the \code{data} input.}

\item{countFilterThreshold}{Numeric; if the number of parent cells
 is less than this threshold, we remove that
file. Default is 0, which means filter is disabled.}
}
\value{
A \code{COMPASSContainer} returns a list made up of the same
components as input the model, but checks and sanitizes the supplied data
to ensure that it conforms to the expectations outlied above.
}
\description{
This function generates the data container suitable for use with
\code{COMPASS}.
}
\details{
The \code{names} attributes for the \code{data} and \code{counts}
objects passed should match.
}
\examples{
set.seed(123)
n <- 10 ## number of samples
k <- 3 ## number of markers

## generate some sample data
sid_vec <- paste0("sid_", 1:n) ## sample ids; unique names used to denote samples
iid_vec <- rep_len( paste0("iid_", 1:(n/2) ), n ) ## individual ids

## generate n matrices of 'cell intensities'
data <- replicate(n, {
  nrow <- round(runif(1) * 1E2 + 1000)
  ncol <- k
  vals <- rexp( nrow * ncol, runif(1, 1E-5, 1E-3) )
  vals[ vals < 2000 ] <- 0
  output <- matrix(vals, nrow, ncol)
  output <- output[ apply(output, 1, sum) > 0, ]
  colnames(output) <- paste0("M", 1:k)
  return(output)
})
names(data) <- sid_vec

## make a sample metadata data.frame
meta <- data.frame(
  sid=sid_vec,
  iid=iid_vec,
  trt=rep( c("Control", "Treatment"), each=5 )
)

## generate an example total counts
## recall that cells not expressing any marker are not included
## in the 'data' matrices
counts <- sapply(data, nrow) + round( rnorm(n, 1E3, 1E2) )
counts <- setNames( as.integer(counts), names(counts) )

## insert everything into a COMPASSContainer
CC <- COMPASSContainer(data, counts, meta, "iid", "sid")
}
