% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/AlphaFold.R
\name{af_predictions}
\alias{af_predictions}
\alias{af_prediction_view}
\alias{af_colorfunc_by_position}
\title{AlphaFold Protein Structure Retrieval and Display}
\usage{
af_predictions(uniprot_ids)

af_prediction_view(tbl, bfc = BiocFileCache())

af_colorfunc_by_position(
    tbl,
    pos,
    value,
    pos_max = NULL,
    palette = colorspace::diverging_hcl(11),
    palette_min = NULL,
    palette_max = NULL
)
}
\arguments{
\item{uniprot_ids}{\code{character()} UniProt accession identifiers
(\code{uniprot_id} in AlphaMissense tables).}

\item{tbl}{A tibble containing information on the UniProt protein and
AlphaMissense predicted amino acid effects.

For \code{av_prediction_view()} the tibble must have columns
\code{uniprot_id}, \code{protein_variant}, \code{am_pathogenicity}, and
\code{am_class}, as in tibbles returned by \code{am_data("hg38")} or
\code{am_data("aa_substitutions")}, for instance. The \code{uniprot_id} must
contain a single unique value.

For \code{af_colorfunc_by_position()} the tibble must have columns \code{pos}
and \code{value}, as described below.}

\item{bfc}{An object created with \code{BiocFileCache::BiocFileCache()},
representing the location used to cache PDB files retrieved by
\code{av_prediction_view()}. The default is the BiocFileCache
installation-wide location.}

\item{pos}{the symbol or name of the column in \code{tbl} containing
amino acid residue positions in the protein.}

\item{value}{the symbol or name of the column in \code{tbl} containing
values to be used for coloring amino acid residues in the
protein.}

\item{pos_max}{integer(1) the maximum residue position in the
protein to be visualized. Default: the maximum value in \code{pos}.}

\item{palette}{character() vector of colors to be used in
visualization. The default (\code{colorspace::diverging_hcl(11)})
produces colors ranging from blue (low) to red (high).}

\item{palette_min}{numeric(1) the value bounding the minimum
palette color. The default is the minimum of \code{value}; a common
value when plotting pathogenicity might be \code{0}.}

\item{palette_max}{numeric(1) the value bounding the maximum
palette color. The default is the maximum of \code{value}; a common
value when plotting pathogenicity might be \code{1}.}
}
\value{
\code{af_predictions()} returns a tibble. Each row represents the
AlphaFold prediction associated with the corresponding uniprot
accession. Columns include:
\itemize{
\item \code{entryId}: AlphaFold identifier.
\item \code{gene}: gene symbol corresponding to UniProt protein.
\item \code{uniprotAccession}, \code{uniprotId}, \code{uniprotDescription}:
UniProt characterization. AlphaMissense's \code{uniprot_id} is
AlphaFold's \code{uniprotAccession}.
\item \code{taxId}, \code{organismScientificName}: Organism information.
\item \code{uniprotStart}, \code{uniprotEnd}, \code{uniprotSequence}:
protein sequence information.
\item \code{modelCreatedDate}, \code{latestVersion}, \code{allVersions},
\code{isReviewed}, \code{isReferenceProteome}: AlphaFold provenance
information.
\item \code{cifUrl}, \code{bcifUrl}, \code{pdbUrl}:
URLs to AlphaFold 3-dimensional molecular representations.
\item \code{paeImageUrl}, \code{paeDocUrl}: 'Predicted Aligned Error' heat map
and underlying data. These can be used to assess the confidence
in relative orientation of residues in different domains, as
described in part in the AlphaFold FAQ
\url{https://alphafold.ebi.ac.uk/faq}
}

\code{af_prediction_view()} displays an interactive view of the protein
in an RStudio panel or browser tab.

\code{af_colorfunc_by_position()} returns a character(1) vector
representation of the Javascript function, with color vector
injected.
}
\description{
\code{af_predictions()} retrieves information about
AlphaFold predictions associated with UniProt accession
identifiers.

\code{af_prediction_view()} summarizes effects of possible
amino acid changes in a single UniProt protein. The changes are
displayed on the AlphaFold-predicted structure.

\code{af_colorfunc_by_position()} generates a Javascript
function to be used in \code{rd3mol::m_set_style()} to color
residues by position, e.g., when visualizing median predicted
pathogenicity.
}
\details{
\code{af_predictions()} queries the \code{prediction} endpoint of the
AlphaFold API described at \url{https://alphafold.ebi.ac.uk/api-docs}.

\code{af_prediction_view()} uses \code{tbl} to calculate median pathogenicity
at each amino acid position, using
\code{am_aa_pathogenicity()}. Predicted protein structure is retrieved
from the unique \code{uniprot_id} using \code{af_predictions()} and the
\code{pdbUrl} returned by that function. Protein structure is visualized
using the r3dmol \url{https://cran.R-project.org/package=r3dmol}
package. Amino acids are colored using \code{aa_pathogenicity_median}
and \code{af_colorfunc_by_position()} with default palette defined on
the interval 0, 1.

\code{af_colorfunc_by_position()} uses a template mechanism to inject
a vector of position-specific colors into a Javascript function
used by \code{r3dmol::m_set_style()} / \code{r3dmol::m_style_cartoon()} to
color residues by position. Positions for which no color is
specified are colored \code{'gray'}. The template can be seen with
\code{AlphaMissenseR:::js_template("colorfunc")}.
}
\examples{

## af_predictions

uniprot_ids <-
    am_data("aa_substitutions") |>
    dplyr::filter(uniprot_id \%like\% "P3555\%") |>
    dplyr::distinct(uniprot_id) |>
    pull(uniprot_id)
af_predictions(uniprot_ids)


## af_prediction_view()

P35557 <-
    am_data("hg38") |>
    dplyr::filter(uniprot_id == "P35557")
af_prediction_view(P35557)

## no AlphaFold prediction for this protein
P35555 <-
    am_data("aa_substitutions") |>
    dplyr::filter(uniprot_id == "P35555")
tryCatch({
    af_prediction_view(P35555)
}, error = identity)


## af_colorfunc_by_position()

df <- tibble(
    pos = 1 + 1:10, # no color information for position 1
    value = 10:1 / 10
)
colorfunc <- af_colorfunc_by_position(
    df,
    "pos", "value",
    pos_max = 12    # no color information for position 12
)
cat(colorfunc)

## template used for Javascript function
cat(
    AlphaMissenseR:::js_template("colorfunc", colors = "..."),
    "\n"
)

}
