% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/db.R
\name{db_connect}
\alias{db_connect}
\alias{db_tables}
\alias{db_temporary_table}
\alias{db_range_join}
\alias{db_disconnect}
\alias{db_disconnect_all}
\title{Manipulate the Database of Missense Mutations}
\usage{
db_connect(
    record = ALPHAMISSENSE_RECORD,
    bfc = BiocFileCache(),
    read_only = TRUE,
    managed = read_only
)

db_tables(db = db_connect())

db_temporary_table(db, value, to)

db_range_join(db, key, join, to)

db_disconnect(db = db_connect())

db_disconnect_all()
}
\arguments{
\item{record}{character(1) Zenodo record for the AlphaMissense data
resources.}

\item{bfc}{an object returned by \code{BiocFileCache()} representing the
location where downloaded files and the parsed database will be
stored. The default is the 'global' BiocFileCache.}

\item{read_only}{logical(1) open the connection 'read only'.
\code{TRUE} protects against overwriting existing data and is the
default.}

\item{managed}{logical(1) when \code{TRUE}, re-use an existing managed
connection to the same database.}

\item{db}{\code{duckdb_connection} object, returned by \code{db_connect()}.}

\item{value}{a \code{data.frame} / \code{tibble} containing data to be placed
in a temporary table, e.g., from a GenomicRanges object to be
used in a range join.}

\item{to}{the character(1) name of the table to be created}

\item{key}{a character(1) table name in \code{db} containing missense
mutation coordinates.}

\item{join}{a character(1) table name in \code{db} containing ranges to
be used for joining with (filtering) \code{key}.}
}
\value{
\code{db_connect()} returns an open \code{duckdb_connection} to the
AlphaMissense record-specific database.

\code{db_tables()} returns a character vector of database table
names.

\code{db_temporary_table()} returns the temporary table as a
dbplyr \code{tibble}.

\code{db_range_join()} returns \code{to} (the temporary table created
from the join) as a dbplyr tibble.

\code{db_disconnect()} returns \code{FALSE} if the connection has
already been closed or is not valid (via \code{dbIsValid()}) or
\code{TRUE} if disconnection is successful. Values are returned
invisibly.

\code{db_disconnect_all()} returns the \code{db_disconnect()} value
for each connection, invisibly.
}
\description{
\code{db_connect()} manages connections to AlphaMissense
record-specific databases. By default, connections are created
once and reused.

\code{db_tables()} queries for the names of temporary and
regular tables defined in the database.

\code{db_temporary_table()} creates a temporary (for the
duration of the duckdb connection) table from a tibble.

\code{db_range_join()} performs a range join, finding all
positions in \code{key} within ranges defined by \code{join}. The result
is stored in table \code{to}.

\code{db_disconnect()} disconnects the duckdb database and
shuts down the DuckDB server associated with the
connection. Temporary tables are lost.

\code{db_disconnect_all()} disconnects all managed duckdb
database connection.
}
\details{
For \code{db_connect()}, set \code{managed = FALSE} when, for instance,
accessing a database in a separate process. Remember to capture the
database connection \code{db_unmanaged <- db_connect(managed = FALSE)}
and disconnect when done `db_disconnect(db_unmanaged). Connections
are managed by default.

\code{db_temporary_table()} \strong{overwrites} an existing table
with name \code{to}.

\code{db_range_join()} \strong{overwrites} an existing table \code{to}.
The table \code{key} is usually \code{"hg19"} or \code{"hg38"} and must have
\code{CHROM} and \code{POS} columns. The table \code{join} must have columns
\code{CHROM}, \code{start} and \code{end}. Following \emph{Bioconductor}
convention and as reported in \code{am_browse()}, coordinates are
1-based and ranges defined by \code{start} and \code{end} are closed. All
columns from both \code{key} and \code{join} are included, so column names
(other than \code{CHROM}) cannot be duplicated.

\code{db_disconnect()} should be called on each unmanaged
connection, and once (to free the default managed connection)
at the end of a session.
}
\examples{
db_connect()          # default 'read-only' connection

db_rw <- db_connect(read_only = FALSE)

am_data("hg38")       # uses the default, 'read-only' connection
db_tables()           # connections initially share the same tables
db_tables(db_rw)

## ranges of interest -- the first 200000 bases on chromsomes 1-4.
ranges <- tibble(
    CHROM = paste0("chr", 1:4),
    start = rep(1, 4),
    end = rep(200000, 4)
)
db_temporary_table(db_rw, ranges, "ranges")

db_tables(db_rw)      # temporary table available to the db_rw connection...
db_tables()           # ...but not to the read-only connection

rng <- db_range_join(db_rw, "hg38", "ranges", "ranges_overlaps")
rng
rng |>
    count(CHROM) |>
    arrange(CHROM)

db_disconnect(db_rw)  # explicit read-write connection
db_disconnect()       # implicit read-only connection

db_disconnect_all()

}
